#!/usr/bin/python

"""Module to handle semantic interface to SSH configuration files.
   Not as fancy as it might be, but it should be fairly straightforward.
   
   BUG: For simplicity's sake, this uses a lot of linear searches where it 
   should use dictionaries."""

import parsely

# To initialize this module, we first load the grammar.
_ssh_fmt = parsely.loadFormat('ssh_config')
_ssh_ts  = _ssh_fmt.getTypeSystem()

_validOptions = ( 'BatchMode', 'Cipher', 'ClearAllForwardings',
		  'Compression', 'CompressionLevel',
		  'ConnectionAttempts', 'EscapeChar', 'FallBackToRsh',
		  'ForwardAgent', 'ForwardX11', 'GatewayPorts',
		  'GlobalKnownHostsFile', 'HostName', 'IdentityFile',
		  'KeepAlive', 'KerberosAuthentication',
		  'KerberosTgtPassing', 'LocalForward',
		  'NumberOfPasswordPrompts', 'PasswordAuthentication',
		  'PasswordPromptHost', 'PasswordPromptLogin', 'Port',
		  'ProxyCommand', 'RemoteForward',
		  'RhostsAuthentication', 'RhostsRSAAuthentication',
		  'RSAAuthentication', 'StrictHostKeyChecking',
		  'TISAuthentication', 'UsePrivilegedPort', 'User',
		  'UserKnownHostsFile', 'UseRsh', 'XAuthLocation')

class SSHConfigFile:
    def __init__(self, filename):
	self.filename = filename
	try:
	    self.file = _ssh_fmt.parseFile(filename)
	except Exception, e:
	    raise Exception("Error while trying to read %s: %s" % (filename,
								   e))

	self.hosts = parsely.tree.buildIndex(self.file, 
					     pathToNodes='hosts.*',
					     indexPath='hostDecl.host',
					     unique=1)

    def getHosts(self):
	return self.hosts.keys() 

    def getHostOption(self, hostName, optionName):
	"""Gets the value of the option optionName associated with the host
	   hostName.  (hostName may be '*' or None)."""

        decl = self.__findOptionDecl(hostName, optionName)

	if decl is not None:
	    return _optionVal(decl.v)
	else:
	    return None

    def setHostOption(self, hostName, optionName, val):

	decl = self.__findOptionDecl(hostName, optionName)

	if decl is not None:
	    decl.v = val
	else:
	    optionDecl = _ssh_ts.newNode('Option:single', 
					 (optionName, str(val), '\n'), '')
	    if hostName is None:
		self.file.initial.append(optionDecl)
	    elif self.hosts.has_key(hostName):
		self.hosts[hostName].body.append(optionDecl)
	    else:
		hostDecl = _ssh_ts.newNode('HostDecl',
					   ('Host', hostName, '\n'), '')

		section = _ssh_ts.newNode('Section')
		section.hostDecl = hostDecl
		section.body.append(optionDecl)

		self.file.hosts.append(section)

		self.hosts[hostName] = section

    def __findOptionDecl(self, hostName, optionName):
        if hostName is None:
	    optionBlock = self.file.initial
	elif self.hosts.has_key(hostName):
	    optionBlock = self.hosts[hostName].body
	else:
	    return None

	for optionDecl in optionBlock:
	    if str(optionDecl.k) == optionName:
		return optionDecl
	
	return None

    def flush(self):
	self.file.flush()
	
    def optionsUsedForHost(self, hostName):
	if hostName is None:
	    optionBlock = self.file.initial
	elif self.hosts.has_key(hostName):
	    optionBlock = self.hosts[hostName].body
	else:
	    return []
	
	lst = []
	for optionDecl in optionBlock:
	    lst.append( str(optionDecl.k) )

	return lst
	    

def _optionVal(node):
    lst = map(str, node)
    if len(lst) == 1:
	return lst[0]
    else:
	return lst

if __name__ == '__main__':
    import sys
    if len(sys.argv) > 0 and sys.argv[-1][-3:] != '.py':
	fname = sys.argv[-1]
    else:
	fname = '/etc/ssh/ssh_config'

    conf = SSHConfigFile(fname)

    for host in conf.getHosts():
	print "Host", host
	for option in conf.optionsUsedForHost(host):
	    print "  ", option, "=", conf.getHostOption(host,option)

    #conf.setHostOption('Foob', 'bar', 'baz')
    #conf.setHostOption('*', 'quux', 'quum')
    #conf.setHostOption(None, 'quum', 'quum')
    #print conf.file.dump()
