/* Parsely - A cross-language tool for parsing and file manipulation.
 *
 * Copyright (C) 1999-2000 Nick Mathewson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _PARSELY_H
#define _PARSELY_H

#ifndef __GLIB_H__
#include <glib.h>
#endif

typedef GNode PrNode;
typedef GSList PrPath;
typedef gboolean (*PrNodePredicate) (PrNode *node,
				     gpointer user_data);

typedef struct _PrNodeType PrNodeType;
typedef struct _PrTypeSystem PrTypeSystem;
typedef enum _PrNodeCode PrNodeCode;

enum _PrNodeCode {
	PR_LEAF,
	PR_LIST,
	PR_STRUCT,
	PR_EXLIST,
	PR_INCLUDE,
	PR_FILE,
};

struct _PrNodeType {
	/*== Public ==*/
	PrNodeCode code;
	gchar* typeName;
	gchar* fullName;
	gchar* variantName;

	/*== Private ==*/
	PrTypeSystem *typeSystem;

	union {
		struct {
		        gchar* defaultVal;
			gchar* defaultSpace;
		} leafData;
		struct {
			gint nMembers;
			GSList* memberTypes;
			GSList* tagNames;
		} structData;
		struct {
			gchar* baseType;
			gchar* termType;
			gchar* sepType;
			PrNode* defaultSeparator;
			gboolean isExclusive;
		} listData;
	} data;
};

struct _PrTypeSystem {
	GHashTable* typesByName;
	GHashTable* typesByLongName;
	gchar* defaultSpace;
};

/****
 * Navigation methods
 ****/
PrNode*  pr_get_path(PrNode* base, const gchar* path);
gboolean pr_set_path(PrNode* base, const gchar* path, PrNode* val);
PrNode*  pr_extract_path(PrNode* base, const gchar* path);
gboolean pr_delete_path(PrNode* base, const gchar* path);

GSList*  pr_get_path_list(PrNode* base, const gchar* path);

PrNode*  pr_get_tag(PrNode* base, const gchar* tag);
gboolean pr_set_tag(PrNode* base, const gchar* tag, PrNode* val);
PrNode*  pr_get_idx(PrNode* base, gint idx);
gboolean pr_set_idx(PrNode* base, gint idx, PrNode* val);

/****
 * Path-based methods
 ****/
PrPath*  pr_parse_path(const gchar*);
void     pr_free_path(PrPath*);

PrNode*  pr_get_ppath(PrNode* base, const PrPath* path);
gboolean pr_set_ppath(PrNode* base, const PrPath* path, PrNode* val);
PrNode*  pr_extract_ppath(PrNode* base, const PrPath* path);
gboolean pr_delete_ppath(PrNode* base, const PrPath* path);

GSList*  pr_get_ppath_list(PrNode* base, const PrPath* path);

/****
 * Node manipulation methods
 ****/
const gchar* pr_leaf_get_trailing_space(PrNode* leaf);
const gchar* pr_leaf_get_val(PrNode* val);
void         pr_leaf_set_val(PrNode* leaf, const gchar* val);
void         pr_leaf_set_trailing_space(PrNode* leaf, const gchar* val);

gchar*       pr_node_to_string(PrNode* node);
gboolean     pr_node_is_leaf(PrNode* node);

PrNode*      pr_new_leaf_node(PrNodeType* type,
			 const gchar* val, 
			 const gchar* trailing_space);
PrNode*      pr_new_list_node(PrNodeType* type, ...); /* Null-terminated */
PrNode*      pr_new_struct_node(PrNodeType* type, ...); /* Null-terminated */

PrNode*      pr_new_node_default(PrNodeType* type, gint depth);
PrNode*      pr_new_null_node();

void         pr_node_free(PrNode* node);

/****
 * TypeSystem functions
 ****/

PrNodeType* pr_typesystem_get_type(const PrTypeSystem*, const gchar* typename);

/****
 * Indexing
 ****/

GHashTable* pr_build_index(PrNode* base,
			   const gchar* path_to_nodes,
			   const gchar* path_to_idx,
			   gboolean descend,
			   PrNodePredicate filter,
			   gpointer filter_data,
			   gboolean unique,
			   GHashTable* hash_in);

GHashTable* pr_build_index_ppath(PrNode* base,
				 const PrPath* path_to_nodes,
				 const PrPath* path_to_idx,
				 gboolean descend,
				 PrNodePredicate filter,
				 gpointer filter_data,
				 gboolean unique,
				 GHashTable* hash_in);

/****
 * Traversal
 ****/

/* XXXX Write */

/****
 * Used only in initialization functions.
 ****/
PrTypeSystem* pr_typesystem_new(const gchar* defaultSpace);
void          pr_typesystem_add_type(PrTypeSystem *typesys, 
				     PrNodeType* type);
void          pr_typesystem_free(PrTypeSystem* typeSystem);

#endif /* _PARSELY_H */
