/* Parsely - A cross-language tool for parsing and file manipulation.
 *
 * Copyright (C) 1999-2000 Nick Mathewson
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <glib.h>
#include "parsely.h"

typedef struct _PrPathElement {
	enum PR_PATH_CODE { IDX_PATH, TAG_PATH, STAR_PATH } code;
	union {
		guint idx;
		gchar* tag;
	} val;
} PrPathElement;

typedef struct _PrNodeData {
	PrNodeCode code;
	PrNodeType *type;
	union {
		struct {
			gchar* val;
			gchar* trailing_space;
		} leafData;
		struct {
			gchar* fileName;
			gchar* initialSpace;
			gchar* defaultSpace;
		} fileData;
		struct {
			gboolean isSep;
		} exlistData;
	} data;
} PrNodeData;

/****
 * Helpers
 ****/

#define PR_NODE_DATA(node) ( (PrNodeData*) (node)->data )
#define CODE_IS_BRANCH(code)       \
         (((code) == PR_STRUCT) || \
	  ((code) == PR_LIST) ||   \
          ((code) == PR_EXLIST))
#define CODE_IS_DISPATCH(code)     \
         (((code) == PR_FILE) ||   \
          ((code) == PR_INCLUDE))
#define CODE_IS_LIST(code)         \
         (((code) == PR_LIST) ||   \
          ((code) == PR_EXLIST))

#define NODE_IS_BRANCH(node)                          \
         ((node) && ((node)->data) &&                 \
	  CODE_IS_BRANCH(PR_NODE_DATA(node)->code))
#define NODE_IS_DISPATCH(node)                        \
         ((node) && ((node)->data) &&                 \
	  CODE_IS_DISPATCH(PR_NODE_DATA(node)->code))
#define NODE_IS_LIST(node)                            \
         ((node) && ((node)->data) &&                 \
	  CODE_IS_LIST(PR_NODE_DATA(node)->code))

#define APPEND_SLIST(start,last,data)                   \
 G_STMT_START{                                          \
	if (!start) {                                   \
		start=last=g_slist_prepend(NULL,data);  \
	} else {                                        \
		last=g_slist_append(last,data)->next;   \
	}                                               \
 }G_STMT_END


static GNode*   g_node_insert_after(GNode* parent, 
				     GNode* sibling,
				     GNode* node);

static gint     pr_get_idx_for_tag(const PrNodeType* type,
				   const gchar* tag);

static void     pr_replace_node(PrNode* orig, PrNode* replacement);

static gboolean pr_free_node_data(GNode* _node, gpointer user_data);

static PrNode*  pr_get_single_path_el(PrNode* node, PrPathElement* elem);

static PrNode*  pr_chase_node(PrNode* node);

static GSList*  pr_filter_node_list(GSList* list,
				    PrNodePredicate filter,
				    gpointer filter_data);

static PrNode*  pr_new_struct_node_default(PrNodeType* type, gint depth);

static void     pr_hash_table_insert(GHashTable *hash,
				     gchar *idx,
				     PrNode *node,
				     gboolean unique);

static gboolean pr_node_to_string_helper(GNode* _node, gpointer user_data);

static GNode*
g_node_insert_after(GNode* parent, GNode* sibling, GNode* node)
{
	g_return_val_if_fail(parent != NULL, node);
	g_return_val_if_fail(node != NULL, node);
	g_return_val_if_fail(G_NODE_IS_ROOT(node), node);
	if (sibling)
		g_return_val_if_fail(sibling->parent == parent, node);

	node->parent = parent;
	if (sibling) {
		if (sibling->next) {
			node->next = sibling->next;
			if (node->next)
				node->next->prev = node;
			node->prev = sibling;
			sibling->prev = node;
		}
	} else {
		g_node_append(parent, node);
	}
	return node;
}

static gint
pr_get_idx_for_tag(const PrNodeType* type, const gchar* tag)
{
	GSList* tagList;
	int i;

	g_assert(type->code == PR_STRUCT);

	tagList = type->data.structData.tagNames;
	for(i=0; tagList; tagList = tagList->next, ++i) {
		if (strcmp((gchar*) tagList->data, tag) == 0)
			return i;
	}

	return -1;
}

static void
pr_replace_node(PrNode* orig, PrNode* replacement)
     /* Destroys 'replacement' in the process.
      */
{
	PrNode* children_orig;
	PrNodeData* data_orig;

	if (! replacement) {
		replacement = g_node_new(NULL);
	}

	children_orig = orig->children;
	data_orig = PR_NODE_DATA(orig);
	
	orig->data = replacement->data;
	orig->children = replacement->children;

	replacement->children = children_orig;
	replacement->data = data_orig;
	
	pr_node_free(replacement);
}

static gboolean
pr_free_node_data(GNode* _node, gpointer user_data)
{
	PrNode* node = (PrNode*) _node;
	PrNodeData* data = PR_NODE_DATA(node);
	switch(data->code) {
	case PR_LEAF:
		g_free(data->data.leafData.val);
		g_free(data->data.leafData.trailing_space);
		break;
	case PR_INCLUDE:
		/* XXXX */
		break;
	case PR_FILE:
		g_free(data->data.fileData.fileName);
		g_free(data->data.fileData.initialSpace);
		g_free(data->data.fileData.defaultSpace);
		break;
	case PR_EXLIST:
	case PR_LIST:
	case PR_STRUCT:
		/* Do Nothing. */
		break;
	default:
		g_assert_not_reached();
	}
	
	return FALSE; /* Don't halt traversal. */
}

static PrNode*
pr_get_single_path_el(PrNode* node, PrPathElement* elem)
{
	switch(elem->code) {
	case IDX_PATH:
		return pr_get_idx(node, elem->val.idx);
	case TAG_PATH:
		return pr_get_tag(node, elem->val.tag);
	case STAR_PATH:
		g_return_val_if_fail(0, NULL);
	}

	g_assert_not_reached();
	return NULL;
}

static PrNode*
pr_chase_node(PrNode* node)
{
	PrNodeData* data;

	while (TRUE) {
		g_assert(node);
		data = PR_NODE_DATA(node);
		g_assert(data);

		switch(data->code) {
		case PR_FILE:
		case PR_INCLUDE:
			node = g_node_first_child(node);
			break;
		default:
			return node;
		}
	}
	g_assert_not_reached();
}

static GSList*
pr_filter_node_list(GSList *list,
		    PrNodePredicate filter,
		    gpointer filter_data)
{
	GSList *ret_list, *list_end, *next;

	if (! filter)
		return list;
	
	ret_list = list_end = 0;
	while(list) {
		next = list->next;
		if (filter((PrNode*)list->data, filter_data))
			APPEND_SLIST(ret_list, list_end, list->data);
		else
			g_slist_free(list);

		list = next;
	}
	
	return ret_list;
}

/****
 * Navigation methods
 ****/

PrNode*
pr_get_path(PrNode* base, const gchar* path)
{
	PrPath* parsed_path;
	PrNode* result;

	g_return_val_if_fail(base != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	parsed_path = pr_parse_path(path);
	if (! parsed_path)
		return 0;
	result = pr_get_ppath(base, parsed_path);

	pr_free_path(parsed_path);

	return result;
}

gboolean
pr_set_path(PrNode* base, const gchar* path, PrNode* val)
{
	PrPath* parsed_path;
	gboolean changed;

	g_return_val_if_fail(base != 0, FALSE);
	g_return_val_if_fail(path != 0, FALSE);

	parsed_path = pr_parse_path(path);
	if (! parsed_path)
		return FALSE;

	changed = pr_set_ppath(base, parsed_path, val);
	pr_free_path(parsed_path);

	return changed;
}

PrNode*
pr_extract_path(PrNode* base, const gchar* path)
{
	PrPath* parsed_path;
	PrNode* result;

	g_return_val_if_fail(base != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	parsed_path = pr_parse_path(path);
	if (! parsed_path)
		return 0;
	result = pr_extract_ppath(base, parsed_path);

	pr_free_path(parsed_path);

	return result;
}

gboolean
pr_delete_path(PrNode* base, const gchar* path)
{
	PrPath* parsed_path;
	gboolean changed;

	g_return_val_if_fail(base != NULL, 0);
	g_return_val_if_fail(path != NULL, 0);

	parsed_path = pr_parse_path(path);
	if (! parsed_path)
		return 0;
	changed = pr_delete_ppath(base, parsed_path);
	pr_free_path(parsed_path);
	return changed;
}

GSList*
pr_get_path_list(PrNode* base, const gchar* path)
{
	PrPath* parsed_path;
	GSList* result;

	g_return_val_if_fail(base != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	parsed_path = pr_parse_path(path);
	if (! parsed_path)
		return 0;
	result = pr_get_ppath_list(base, parsed_path);
	pr_free_path(parsed_path);
	return result;
}

PrNode*
pr_get_tag(PrNode* base, const gchar* tag)
{
	PrNodeData* data;
	gint idx;

	g_return_val_if_fail(base != NULL, NULL);
	g_return_val_if_fail(tag != NULL, NULL);

	base = pr_chase_node(base);
	data = PR_NODE_DATA(base);

	g_return_val_if_fail(data->code == PR_STRUCT, NULL);
	g_return_val_if_fail(data->type, NULL);
	
	idx = pr_get_idx_for_tag(data->type, tag);
	g_return_val_if_fail(idx >= 0, NULL);
	
	return (PrNode*) g_node_nth_child(base, idx);
}

gboolean
pr_set_tag(PrNode* base, const gchar* tag, PrNode* val)
{
	PrNodeData* data;
	gint idx;
	GNode* child_node;

	g_return_val_if_fail(base != NULL, FALSE);
	g_return_val_if_fail(tag != NULL, FALSE);
	base = pr_chase_node(base);
	data = PR_NODE_DATA(base);

	g_return_val_if_fail(data->code == PR_STRUCT, FALSE);

	g_assert(G_NODE_IS_ROOT(val));
	g_return_val_if_fail(data->type, FALSE);
	
	idx = pr_get_idx_for_tag(data->type, tag);
	
	g_return_val_if_fail(idx >= 0, FALSE);
	
	child_node = g_node_nth_child(base, idx);
	g_assert(child_node);
	
	pr_replace_node(child_node, val);

	return TRUE;
}

PrNode*
pr_get_idx(PrNode* base, gint idx)
{
	PrNodeData* data;

	g_return_val_if_fail(base != NULL, NULL);
	g_return_val_if_fail(idx >= 0, NULL);

	base = pr_chase_node(base);
	data = PR_NODE_DATA(base);

	g_return_val_if_fail(CODE_IS_BRANCH(data->code), NULL);
	if (data->code == PR_EXLIST)
		idx *= 2;

	return g_node_nth_child(base,idx);
}

gboolean
pr_set_idx(PrNode* base, gint idx, PrNode* val)
{
	PrNodeData* data;
	PrNode* child;

	g_return_val_if_fail(base != NULL, FALSE);
	g_return_val_if_fail(idx >= 0, FALSE);

	base = pr_chase_node(base);
	data = PR_NODE_DATA(base);

	g_return_val_if_fail(CODE_IS_BRANCH(data->code), FALSE);

	if (data->code == PR_EXLIST)
		idx *= 2;
	
	child = g_node_nth_child(base,idx);
	g_return_val_if_fail(child != NULL, FALSE);
	pr_replace_node(child, val);
	return TRUE;
}

/****
 * Parsed-path methods.
 ****/
PrPath*
pr_parse_path(const gchar* path)
{
	guint component_start = 0;
	guint idx = 0;
	gboolean seen_nonint = FALSE;
	GSList *ppath = NULL;
	GSList *path_end = NULL;	
	PrPathElement *cur_element = NULL;	
	gchar ch;
	guint scanned_num = 0;

	do {
		ch = path[idx];
		if (!ch && !ppath) {
			return ppath; 
		} else if (ch == '.' || !ch) {
			g_return_val_if_fail(component_start < idx, NULL);
			cur_element = g_new(PrPathElement, 1);

			if (! seen_nonint) {
				cur_element->code = IDX_PATH;
				cur_element->val.idx = scanned_num;
			} else {
				cur_element->val.tag =
					g_strndup(&path[component_start],
						  idx-component_start);
				if (! strcmp(cur_element->val.tag, "*")) {
					g_free(cur_element->val.tag);
					cur_element->code = STAR_PATH;
				} else {
					cur_element->code = TAG_PATH;
				}
			}

			APPEND_SLIST(ppath,path_end,cur_element);
				
			component_start = idx+1;
			seen_nonint = FALSE;
			scanned_num = 0;
		} else if (!seen_nonint && isdigit(ch)) {
			scanned_num *= 10;
			scanned_num += ch - '0';
		} else {
			seen_nonint = TRUE;
		}
		idx++;
	} while (ch);
		
	return ppath;
}

void pr_free_path(PrPath* path)
{
	PrPathElement *el;
	GSList *next;

	while(path) {
		next = path->next;
		el = (PrPathElement*) path->data;
		if (el->code == TAG_PATH)
			g_free(el->val.tag);
		g_free(el);
		g_slist_free(path);
		path = next;
	}
}

PrNode*
pr_get_ppath(PrNode* base, const PrPath* path)
{
	while(path) {
		g_return_val_if_fail(base != NULL, NULL);
		base = pr_get_single_path_el(base,
					     (PrPathElement*)path->data);
		path = path->next;
	}
	return base;
}

gboolean
pr_set_ppath(PrNode* base, const PrPath* path, PrNode* val)
{
	PrNode* orig;

	orig = pr_get_ppath(base, path);
	g_return_val_if_fail(orig, FALSE);
	
	pr_replace_node(orig, val);
	return TRUE;
}

PrNode*
pr_extract_ppath(PrNode* base, const PrPath* path)
{
	PrNode* orig;
	PrNodeData* parent_data;
	PrNode* next_sep;

	orig = pr_get_ppath(base, path);
	g_return_val_if_fail(orig != NULL, FALSE);
	g_return_val_if_fail(NODE_IS_LIST(orig->parent), FALSE);
	
	/* XXXX Dispatch case? */

	parent_data = PR_NODE_DATA(orig->parent);
	if (parent_data->code == PR_EXLIST) {
		next_sep = orig->next;
		pr_node_free(next_sep);
	}

	g_node_unlink(orig);
	return orig;
}

gboolean
pr_delete_ppath(PrNode* base, const PrPath* path)
{
	PrNode* node;

	node = pr_extract_ppath(base, path);
	g_return_val_if_fail(node != NULL, FALSE);
	pr_node_free(node);

	return TRUE;
}

GSList* pr_get_ppath_list(PrNode* base, const PrPath* path)
{
	PrPathElement *elem;
	GSList *cur_level;
	GSList *lst_idx;
	GSList *tmp;
	GNode *cur_node;
	GNode *node_idx;
	GSList *next_level = NULL;
	GSList *next_level_last = NULL;

	cur_level = g_slist_prepend(cur_level, base);

	while(path && cur_level) {
		elem = (PrPathElement*)path->data;
		for (lst_idx = cur_level; lst_idx; lst_idx = lst_idx->next) {
			cur_node = (GNode*) lst_idx->data;
			if (elem->code == STAR_PATH) {
				node_idx = cur_node->children;
				while(node_idx) {
					APPEND_SLIST(next_level,
						     next_level_last,
						     node_idx);
					node_idx = node_idx->next;
				}
			} else {
				node_idx =pr_get_single_path_el(cur_node,elem);
				APPEND_SLIST(next_level,
					     next_level_last,
					     node_idx);
			}
		}		

		for(lst_idx = cur_level; lst_idx; lst_idx=tmp) {
			tmp = lst_idx->next;
			g_slist_free(lst_idx);
		}

		cur_level = next_level;
		next_level = next_level_last = NULL;
		path = path->next;
	}
	return cur_level;
}

/****
 * Node manipulation methods
 ****/

const gchar*
pr_leaf_get_trailing_space(PrNode* leaf)
{
	PrNodeData* data;

	g_return_val_if_fail(leaf != NULL, NULL);
	
	data = PR_NODE_DATA(leaf);
	g_return_val_if_fail(data->code == PR_LEAF, NULL);

	return data->data.leafData.trailing_space;
}

const gchar*
pr_leaf_get_val(PrNode* leaf)
{
	PrNodeData* data;

	g_return_val_if_fail(leaf != NULL, NULL);
	
	data = PR_NODE_DATA(leaf);
	g_return_val_if_fail(data->code == PR_LEAF, NULL);

	return data->data.leafData.val;
}

void
pr_leaf_set_val(PrNode* leaf, const gchar* val)
{
	PrNodeData* data;

	g_return_if_fail(leaf != NULL);
	g_return_if_fail(val != NULL);	

	data = PR_NODE_DATA(leaf);
	g_return_if_fail(data->code == PR_LEAF);

	g_free(data->data.leafData.val);
	data->data.leafData.val = g_strdup(val);
}

void
pr_leaf_set_trailing_space(PrNode* leaf, const gchar* val)
{
	PrNodeData* data;

	g_return_if_fail(leaf != NULL);
	g_return_if_fail(val != NULL);	

	data = PR_NODE_DATA(leaf);
	g_return_if_fail(data->code == PR_LEAF);

	g_free(data->data.leafData.trailing_space);
	data->data.leafData.trailing_space = g_strdup(val);
}

static gboolean
pr_node_to_string_helper(GNode* _node, gpointer user_data)
{
	PrNode* node = (PrNode*) _node;
	GString* str = (GString*) user_data;

	PrNodeData* data = PR_NODE_DATA(node);

	if (data) {
		g_assert(data->code == PR_LEAF);
		g_string_append(str, data->data.leafData.val);
		g_string_append(str, data->data.leafData.trailing_space);
	}

	return FALSE;
}

gchar*
pr_node_to_string(PrNode* node)
{
	GString* result = g_string_new(NULL);
	gchar* str;
	
	/* INCLUDE??? XXXX */
	g_node_traverse(node, G_IN_ORDER, G_TRAVERSE_LEAFS, -1,
			pr_node_to_string_helper, result);

	str = result->str;
	g_string_free(result,0);
	return str;
}

gboolean
pr_node_is_leaf(PrNode* node)
{
	PrNodeData *data;
	
	if (!node) return FALSE;
	data = PR_NODE_DATA(node);
	if (!data) return FALSE; 
	return (data->code) == PR_LEAF;
}

PrNode*
pr_new_leaf_node(PrNodeType* type,
		 const gchar* val,
		 const gchar* trailing_space)
{
	PrNodeData *data;
	PrNode* node;

	g_return_val_if_fail(!type || type->code == PR_LEAF, NULL);

	if (type && !val)
		val = ((PrNodeType*)type)->data.leafData.defaultVal;
	if (type && !trailing_space)
		trailing_space = 
			((PrNodeType*)type)->data.leafData.defaultSpace;

	g_return_val_if_fail(val != NULL, NULL);
	g_return_val_if_fail(trailing_space != NULL, NULL);
	
	data = g_new(PrNodeData, 1);
	data->code = PR_LEAF;
	data->type = type;
	data->data.leafData.val = g_strdup(val);
	data->data.leafData.trailing_space = g_strdup(trailing_space);
	
	node = g_node_new(data);
	return node;
}

PrNode*
pr_new_list_node(PrNodeType* type, ...)
{
	PrNode *node, *child, *last_child;
	PrNodeData *data;
	va_list arg;
	
	g_return_val_if_fail(!type || CODE_IS_LIST(type->code), NULL);
	data = g_new(PrNodeData, 1);
	data->type = (PrNodeType*) type;
	if (type)
		data->code = type->code;
	else 
		data->code = PR_LIST;
	if (data->code == PR_EXLIST) {
		g_assert(type->code == PR_EXLIST);
		data->data.exlistData.isSep = (gboolean)
			(type->data.listData.sepType != FALSE);
	}

	node = g_node_new(data);
	last_child = NULL;
	
	va_start(arg, type);
	while ( (child = va_arg(arg, PrNode*)) ) {
		g_node_insert_after(node, last_child, child);
		if (child)
			last_child = child;
	}
	va_end(va_list);
	
	return node;
}

PrNode*
pr_new_struct_node(PrNodeType* type, ...)
{
	PrNode *node, *child, *last_child;
	PrNodeData *data;
	gint n_children = 0;
	va_list arg;

	g_return_val_if_fail(type && type->code == PR_STRUCT, NULL);
	data = g_new(PrNodeData, 1);
	data->type = type;
	data->code = PR_STRUCT;

	node = g_node_new(data);
	last_child = NULL;
	
	va_start(arg, type);
	while ( (child = va_arg(arg, PrNode*)) ) {
		++n_children;
		g_node_insert_after(node, last_child, child);
		if (child)
			last_child = child;
	}
	va_end(arg);

	g_return_val_if_fail(n_children == type->data.structData.nMembers,
			     node);

	return node;
}

PrNode*
pr_new_node_default(PrNodeType* type, gint depth)
{
	g_return_val_if_fail(type != NULL, NULL);

	if (depth < 0) {
		return g_node_new(NULL);
	}

	switch(type->code) {
	case PR_LIST:
	case PR_EXLIST:
		return pr_new_list_node(type, NULL);
	case PR_LEAF:
		return pr_new_leaf_node(type, NULL, NULL);
	case PR_STRUCT:
		return pr_new_struct_node_default(type, depth);
	default:
		g_assert_not_reached();
	}
	g_assert_not_reached();
	return NULL;
}

static PrNode*
pr_new_struct_node_default(PrNodeType* type, gint depth)
{
	PrNode *node, *last_child, *member;
	PrTypeSystem *typeSystem;
	PrNodeType *memberType;
	GSList *typeName;
	PrNodeData *data = g_new(PrNodeData, 1);

	g_assert(type->code == PR_STRUCT);

	data->code = PR_STRUCT;
	data->type = type;
	node = g_node_new(data);

	typeSystem = type->typeSystem;
	g_assert(typeSystem);
	last_child = NULL;
	typeName = type->data.structData.memberTypes;
	while (typeName) {
		memberType = pr_typesystem_get_type(typeSystem, 
						    (gchar*)typeName->data);
		member = pr_new_node_default(memberType, depth-1);

		g_node_insert_after(node, last_child, member);
		if (member)
			last_child = member;
		typeName = typeName->next;
	}
	return node;
}

void
pr_node_free(PrNode* node)
{
	if (node) {
		g_node_traverse(node, G_PRE_ORDER, G_TRAVERSE_ALL,
				-1, pr_free_node_data, NULL);
		g_node_destroy(node);
	}
}

PrNodeType* 
pr_typesystem_get_type(const PrTypeSystem* typesystem, const gchar* typename)
{
	gpointer data;
	
	g_return_val_if_fail(typename != NULL, NULL);
	
	data = g_hash_table_lookup(typesystem->typesByLongName, typename);
	if (data)
		return (PrNodeType*) data;
	data = g_hash_table_lookup(typesystem->typesByName, typename);
	if (data)
		return (PrNodeType*) ((GSList*) data)->data;
	
	return NULL;	
}

GHashTable*
pr_build_index(PrNode* base,
	       const gchar* path_to_nodes,
	       const gchar* path_to_idx,
	       gboolean descend,
	       PrNodePredicate filter,
	       gpointer filter_data,
	       gboolean unique,
	       GHashTable* hash_in)
{
	PrPath* nodep = NULL;
	PrPath* idxp = NULL;
	GHashTable* result;

	g_return_val_if_fail(path_to_idx != NULL, hash_in);

	if (path_to_nodes)
		nodep = pr_parse_path(path_to_nodes);
	idxp = pr_parse_path(path_to_idx);
	
	result = pr_build_index_ppath(base, nodep, idxp, descend,
				      filter, filter_data,
				      unique, hash_in);

	pr_free_path(idxp);
	pr_free_path(nodep);
	
	return result;
}
	       
GHashTable* 
pr_build_index_ppath(PrNode* base,
		     const PrPath* path_to_nodes,
		     const PrPath* path_to_idx,
		     gboolean descend,
		     PrNodePredicate filter,
		     gpointer filter_data,
		     gboolean unique,
		     GHashTable* hash_in)
{
	GHashTable *hash = hash_in;
	GSList *nodes; /* Nodes matching path. */
	GSList *list_ptr;
	PrNode *idx;
	gchar *idxval;

	if (! hash)
		hash = g_hash_table_new(g_str_hash, g_str_equal);

	nodes = pr_get_ppath_list(base, path_to_nodes);

	/**
	 * Descend logic
	 **/
#if 0 /* XXXX We should traverse instead. */
	if (descend) {		
		list_ptr = nodes;
		nodes = pr_nodes_closure(nodes);
		free_list (list_ptr);
	}
#endif

	/**
	 * Filter logic
	 **/
	nodes = pr_filter_node_list(nodes, filter, filter_data);

	/**
	 * Add-to-hash logic.
	 **/
	for (list_ptr = nodes; list_ptr; list_ptr = list_ptr->next) {
		idx = pr_get_ppath( (PrNode*)list_ptr->data, path_to_idx);
		if (idx) {
			idxval = pr_node_to_string(idx);
			pr_hash_table_insert(hash, idxval, 
					     (PrNode*)list_ptr->data,
					     unique);
		}
	}
	
	return hash;
}

static void
pr_hash_table_insert(GHashTable *hash,
		     gchar *idx,
		     PrNode *node,
		     gboolean unique)
{
	GSList *last_entry_l;
	PrNode *last_entry_n;
	gpointer last_entry;
	last_entry = g_hash_table_lookup(hash, idx);

	if (unique) {
		last_entry_n = (PrNode*) last_entry;
		if (last_entry_n) {
			g_free(idx);
			return;
		} else {			
			g_hash_table_insert(hash, idx, node);
		}
	} else {
		last_entry_l = (GSList*) last_entry;
		if (last_entry_l) {
			g_free(idx);			
			last_entry_l->next =
				g_slist_prepend(last_entry_l->next, node);
		} else {
			g_hash_table_insert(hash, idx, 
					    g_slist_prepend(NULL, node));
		}
	}	
}
	       
/****
 * For initialization functions.
 ****/

PrTypeSystem*
pr_typesystem_new(const gchar* defaultSpace)
{
	PrTypeSystem *typesys;
	
	g_return_val_if_fail(defaultSpace != NULL, NULL);

	typesys = g_new(PrTypeSystem,1);
	typesys->defaultSpace = g_strdup(defaultSpace);
	typesys->typesByName = g_hash_table_new(g_str_hash, g_str_equal);
	typesys->typesByLongName = g_hash_table_new(g_str_hash, g_str_equal);

	return typesys;
}

void 
pr_typesystem_add_type(PrTypeSystem *typesys, PrNodeType* type)
{
	gpointer oldval;

	g_return_if_fail(typesys != NULL);
	g_return_if_fail(type != NULL);
	g_return_if_fail(type->typeSystem == NULL);
	
	/* 1. Link this type to the typesystem. */
	type->typeSystem = typesys;

	/* 2. Add this type to the long name index. */
	/* Sanity check: type->fullName should be unique! */
	oldval = g_hash_table_lookup(typesys->typesByLongName,
				     type->fullName);
	g_assert(! oldval);
	g_hash_table_insert(typesys->typesByLongName, type->fullName, type);
	
	/* 3. Add this type to the short name index. */
	oldval = g_hash_table_lookup(typesys->typesByName, type->typeName);
	if (oldval) {
		/* We already have at least one type of this name;
		   insert this type into the list. */
		((GSList*)oldval)->next =
			g_slist_prepend(((GSList*)oldval)->next, type);
	} else {
		/* This is the first type of this name. */
		g_hash_table_insert(typesys->typesByName,
				    type->typeName,
				    g_slist_prepend(NULL,type));
	}       
}

static void
pr_free_type(PrNodeType *type)
{
	GSList *lst, *next;

	g_free(type->typeName);
	g_free(type->fullName);
	g_free(type->variantName);
	switch(type->code) {
	case PR_LEAF:
		g_free(type->data.leafData.defaultVal);
		g_free(type->data.leafData.defaultSpace);
		break;
	case PR_STRUCT:
		lst = type->data.structData.memberTypes;
		for (; lst; lst = next) {
			next = lst->next;
			g_free(lst->data);
			g_slist_free(lst);
		}
		lst = type->data.structData.tagNames;
		for (; lst; lst = next) {
			next = lst->next;
			g_free(lst->data);
			g_slist_free(lst);
		}
		break;
	case PR_LIST:
	case PR_EXLIST:
		g_free(type->data.listData.baseType);
		g_free(type->data.listData.termType);
		g_free(type->data.listData.sepType);
		pr_node_free(type->data.listData.defaultSeparator);
		break;
	default:
		g_assert_not_reached();
	}
}


static gboolean
pr_typesystem_free_helper(gpointer key, gpointer val, gpointer data)
{
	gboolean typeByLongName = (GPOINTER_TO_INT(data) != 0);
	GSList *lst, *next;
	
	/* We don't need to free the key; it's just an alias to
	 * the corresponding member of the type struct.
	 */
	
	if (typeByLongName) {
		pr_free_type((PrNodeType*) val);
	} else {
		for (lst = (GSList*) val; lst; lst = next) {
			next = lst->next;
			g_slist_free(lst);
		}
	}
	
	return 1; /* Remove this entry. */
}

void
pr_typesystem_free(PrTypeSystem* typeSystem)
{
	g_free(typeSystem->defaultSpace);
	g_hash_table_foreach_remove(typeSystem->typesByName,
				    pr_typesystem_free_helper,
				    GINT_TO_POINTER(0));
	g_hash_table_foreach_remove(typeSystem->typesByLongName,
				    pr_typesystem_free_helper,
				    GINT_TO_POINTER(1));
}

