#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Utility functions used in parsely code.  These shouldn't be needed to
   by client code."""   

import string, os, posix, re, sys, errno
from types import StringType, TupleType, ListType
import stat
from stat import ST_MODE

from hak_spark import _namelist

####
## List manipulation
####
def _extendList(lst1,lst2):
    """Inserts all the elements of <lst2> at the end of <lst1>.  Serves
       as an alternative to the 'extend' member function, which wasn't
       added until Python 1.5.2.

       Modifies: lst1"""
    l = len(lst1)
    lst1[l:l] = list(lst2)

def _joinLists(*ls):
    """Given a series of lists, returns the concatenation of all those
       lists."""
    res = []
    for l in ls:
	_extendList(res,l)
    return res
            
def _wantList(x):
    """If x is a sequence, returns x.  Otherwise returns a singleton tuple
       containing x."""
    if _isSequence(x):        
        return x
    else:
        return (x,)

def _flattenList(x):
    """If x is sequence of sequences and elements, flattens the sequences.
       Otherwise, returns a singleton tuple containing x.

       For example, _flattenList([1,3,[5, 7],[[9]]]) = [1,3,5,7,[9]]"""
    if _isSequence(x):        
        lOut = []
        for element in x:
            if _isSequence(element):
                _extendList(lOut,element)
            else:
                lOut.append(element)
        return lOut
    else:
        return (x,)
    
def _isSequence(x):
    """Returns true iff x is a tuple or a list."""
    t = type(x)
    return t is TupleType or t is ListType

def _addFormatMember(dict, name, val, unique=None, list=None):
    """Ensures that <dict> has a key <name> pointing to <val>.

       If <unique>, then raises an error if <dict> already has a key <name>.
       If not <unique>, then <dict>[<name>] will map to a list of values,
          of which <val> will be a member.

       If <list>, then <val> will be inserted into <list> iff it was not
       previously a member of <dict>.

       Modifies: <list>, <dict>."""
       
    if dict.has_key(name):
	if unique:
            import parsely.format
	    raise parsely.format.GrammarError("Duplicate key", name)
	else:
	    dict[name].append(val)
    else:
	if unique:
	    dict[name] = val
	else:
	    dict[name] = [ val ]
        if list != None:
            list.append(val)

####
## String manipulation.
####

def _escNonPrinting(s):
    """Returns a new string with the same characters as <s>, except that
       all nonprinting characters will be escaped."""
    l = []
    for c in s:
        if ord(c) >= 32:
            l.append(c)
        elif 7 <= ord(c) <= 13:
            l.append('\\' + "abtnvfr"[ord(c)-7])
        else:
            l.append('\\%.3o' % ord(c))
    return string.join(l,'')

# This regular expression matches the backslashed characters in a string.
_strCharRe = re.compile(
    r'\\\\ | \\[abtnvfr\n] | \\x[A-Fa-f0-9]{2} | \\ [0-7]{1,3}',
    re.X)
_strCharMap = { 'a' : '\a',
                'b' : '\b',
                't' : '\t',
                'n' : '\n',
                'v' : '\v',
                'f' : '\f',
                'r' : '\r',
		'\n' : '\n' }

def __unEscStrChar(match):
    r'''Given a backslashed character representation such as \n or
        \000, returns the actual character.  Used by _unEscStr.'''
    s = match.group(0)
    assert s[0] == '\\'
    if _strCharMap.has_key(s[1]): return _strCharMap[s[1]]
    if s[1] == 'x':
        return chr(atoi(s[2:],16))
    else:
        return chr(atoi(s[1:],8))

def _unEscStr(s):
    '''Replaces all escaped characters in a string with their equivalents.'''
    return _strCharRe.sub(__unEscStrChar,s)

####
## File manipulation
####
__stdDefault = []
def fileContents(name, path=None, default=__stdDefault):
    """Returns the contents of the file <name>.  Raises an appropriate
       exception if the file cannot be read can't be read.

       If <path> is provided, then it searches for <name> in
       every component of the <path>."""
    try:
	f = path_open(path, name)
	contents = f.read()
	f.close()
	return contents

    except Exception, ex:
	if default is not __stdDefault:
	    return default
	else:	    
	    raise ex

def path_open(path, fname):
    """For every element of path, tries to open the file 'element/fname'.
       If path is empty, searches 'fname'.  If no open is successful,
       propagates the last exception.  On success, returns an open
       filehandle."""
    if not path or len(path) ==0:
        path = ("",)
    ex = None
    for pathElement in path:
        try:
            return open(os.path.join(pathElement, fname), 'r')
        except Exception, ex:
            e = ex
    raise e

def replaceFile(fname, contents, mode=-1):
    """Tries to replace the file named by <fname> with <contents>.  The
       mode on the file is set to <mode>, if provided."""
    
    if mode==-1:
        try:
	    mode = posix.stat(fname)[stat.ST_MODE]
	    if stat.S_ISLNK(mode) or stat.S_ISREG(mode):
		pass
	    else:
		raise ParselyException("Tried to replace special file %s" %
				       fname)
            mode = mode & 0777
        except os.error, e: 
	    # If the file didn't exist, feel free to write it.  If we
	    # got any other error, die.
	    if e.args[0] == errno.ENOENT:
		mode = 0644
	    else:
		raise e
    i = 1
    while 1:
        try:
            tmpname = fname + ".tmp_" + str(i)
            # Make a new file; fail if it already exists.
            fd = os.open(tmpname, os.O_CREAT | os.O_WRONLY | os.O_EXCL, mode)
	    break
        except os.error, e:
	    os.close(fd)
	    if e.args[0] != errno.EEXIST:
		raise e
            i = i + 1
    
    written = os.write(fd,contents) 
    if written != len(contents):
	# I don't think this ever happens unless we specify O_NONBLOCK.
	raise "Woah! One of my assumptions just got violated!"
    os.close(fd)
    os.rename(tmpname, fname)

    return 1

####
## Identifier generation
####
class UniqueIDDomain:
    """A UniqueIDDomain maps one kind of unique identifiers (such as
       strings) to another kind of unique identifiers (such as C 
       identifiers).  It guarantees that no two strings in the first
       domain map to a single string in the second domain.
       """

    ##Fields:
    # cache: a map from identifiers to transformed identifiers.
    # used: a set of all the transformed identifiers we're already using.

    def __init__(self):
	self.cache = {}
	self.used = {}
    
    def get(self, id):
	"""Returns the unique transformed identifier for id."""
	transformed = self.cache.get(id,None)
	if transformed:
	    return transformed

	transformed = self._transform(id)
	i = 1
	unique = transformed
	while self.used.has_key(unique):
	    i = i + 1
	    unique = "%s_%d" % (transformed, i)
	self.cache[id] = unique
	self.used[unique] = 1
	return unique

    def _transform(self, id):
	"""Helper function for use in subclasses: performs the actual
	   domain conversion function."""
	return id

####
## Misc
####

def _trimFileFromArgv(fname):
    """A script can be invoked either as 'python ./script args' or as 
       './script args'.  This means that the arguments might start either
       in argv[0] or argv[1].

       Given the name of a script, this function removes the script from
       the arguments list, if it's present."""
    if len(sys.argv) == 0:
        return
    first = sys.argv[0]
    l = len(fname)
    if ((first[-(l+3):] == fname + ".py") or
        (first[-(l+4):] == fname + ".pyo") or
        (first[-(l+4):] == fname + ".pyc")):
        sys.argv = sys.argv[1:]
        
