#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Tree browsing code in GTK.  It is a hack, and meant for developers
   only.  It should prove handy in debugging new grammars, but anybody
   who even thinks of exposing this interface to end users should be
   burned in effigy."""

import gtk
import GDK
import string
import parsely.tree
from parsely._util import _escNonPrinting

def _q(s):
    '''Returns the string <s>, escaped, in quotes.'''
    return '"%s"' % _escNonPrinting(s)

def alignedLabel(text,align='right'):
    """Returns a right-aligned GtkLabel bearing the text <text>."""
    l = gtk.GtkLabel(text)
    if align=='right':
	a = gtk.GtkAlignment(1.00,0.50, 0.0, 0.0)
    else:
	a = gtk.GtkAlignment(0.00,0.50, 0.0, 0.0)
    a.add(l)
    return a

class TypeBrowsingArea:
    ##Fields:
    # typesys
    # clist
    # widget
    # _o


    def __init__(self, typesys):
	self.typesys = typesys
	unames = typesys.typesByLongName.keys()
	unames.sort()
	self.clist = gtk.GtkCList(4, titles=['Name', 'Variant', 
					     'Kind', 'Default'])
	for uname in unames:
	    type = typesys.typesByLongName[uname]
	    name = type.getName()
	    variant = type.variantName
	    if not variant: variant = ""
	    if isinstance(type, parsely.tree.LeafNodeType):
		kind = "Leaf"
		default = _q(str(type.getDefaultNode()))
	    elif isinstance(type, parsely.tree.ListNodeType):
		if type.exclusive:
		    excl = "Exclusive list"
		else:
		    excl = "List"
		if type.sepType:
		    sep = " separated by %s" % type.sepType
		elif type.sepType:
		    sep = " terminated by %s" % type.termType
		else:
		    sep = ""
		kind = "%s of %s%s" % (excl, type.baseType, sep)
		default = _q("")
	    elif isinstance(type, parsely.tree.StructNodeType):
		kind = "Struct: "
		lst = []
		for i in range(len(type.tags)):
		    if type.tags[i]:
			lst.append("%s:%s" % (type.memberTypes[i], 
					      type.tags[i]))
		    else:
			lst.append(type.memberTypes[i])
		kind = string.join(lst)
		default = _q(str(type.getDefaultNode()))
	    self.clist.append((name, variant, kind, default))
	
	self.clist.columns_autosize()
	scrolled = gtk.GtkScrolledWindow()
	scrolled.add(self.clist)
	scrolled.set_usize(400,300)	
	self.widget = scrolled
	self._o = scrolled._o

class BrowsingArea:
    ##Fields: 
    # widget - the widget for this area
    # node - the underlying Node object.
    # _o - the GtkObject for widget
    # tree - the CTree widget
    
    ##Editable only
    # selected_row
    # spaceEnt
    # tree
    # valEnt

    ##Dump only
    # dumpArea
    # dumpWidget
    
    def __init__(self,node, editable, dump=0):
	'''Given a parsely node <node>, returns a scrollable gtkctree to
	   display that node.  If <editable>, allows leaf nodes to be edited.
	   If <dump>, provides a dump '''
	# Make the CTree; add the node's information to it.
	CT= gtk.GtkCTree(4, titles=["Node", "Type", "Value", "Space"])
	self.tree = CT
	CT.set_column_auto_resize(0,1)
	CT.set_column_auto_resize(1,1)
	self.addNodeInfoToCTree(node, "File", None)

	# Make the GtkScrolled object.
	Scrolled = gtk.GtkScrolledWindow()
	Scrolled.add(CT)
	Scrolled.set_usize(400,300)

	if not editable:
	    self.widget = Scrolled
	    self._o = Scrolled._o
	    self.node = node
	    return

	if dump:
	    self.dumpArea = gtk.GtkText()
	    self.dumpArea.set_editable(0)
	    scrol = gtk.GtkScrolledWindow()
	    scrol.add(self.dumpArea)
	else:
	    scrol = None

	# From here on in, we're editable.
	tab = gtk.GtkTable(2,2)
	tab.set_row_spacings(3)
	tab.set_col_spacings(5)

	# Set up the labels and entryies.
	tab.attach(alignedLabel("Value:"),0,1,0,1,gtk.FILL,0,5)
	tab.attach(alignedLabel("Trailing space:"),0,1,1,2,gtk.FILL,0,5)    
	valEntry = gtk.GtkEntry()
	spaceEntry = gtk.GtkEntry()
	valEntry.set_editable(0)
	spaceEntry.set_editable(0)

	self.selected_row = None
	self.valEnt = valEntry
	self.spaceEnt = spaceEntry

	tab.attach(valEntry,1,2,0,1,xpadding=2)
	tab.attach(spaceEntry,1,2,1,2, xpadding=2)

	box = gtk.GtkVBox(3)
	box.set_homogeneous(0)
	box.pack_start(Scrolled, 1, 1)
	box.pack_start(gtk.GtkHSeparator(),0,0)
	box.pack_start(tab, 0, 0)

	if dump:
	    dbox = gtk.GtkVBox(2)
	    dbox.set_homogeneous(0)
	    dbox.pack_start(
		alignedLabel("Node value:", 'left'), 0,0, padding=4 )
	    dbox.pack_start(scrol,1,1)
 
	    paned = gtk.GtkVPaned()
	    paned.add1(box)
	    paned.add2(dbox)
	    box = paned

	CT.connect("select-row", self.onSelect)
	CT.connect("unselect-row", self.onUnselect)
	valEntry.connect("changed", self.onChanged, "V")
	spaceEntry.connect("changed", self.onChanged, "S")
	CT.connect("destroy", self.onTreeDestroy)

	self.widget = box
	self._o = box._o

    def addNodeInfoToCTree(self, node, name, parent):
	'''Adds a a CTree node for the Parsely node 'node' to the
	   CTree tree.  Recurses to the bottom of the tree -- that's a hack,
	   but not a very bad one.  These files are meant to fit easily
	   into memory anyway.'''
        tree = self.tree
	if node is None:
	    isLeaf = 1
	    typeName = "---"
	    val = ""
	    space = ""
	else:
	    isLeaf = node.isLeaf()
	    typeName = node.getType()
	    if typeName:
		typeName = typeName.getLongName()
	    if not typeName:
		typeName = "???"

	    if isLeaf:
		val = _q(node.val)
		space = _q(node.getTrailingSpace())
	    else:
		val = space = ""

	n = tree.insert_node(parent, None,
			     text=[ name, typeName, val, space],
			     is_leaf=isLeaf)

	tree.node_set_row_data(n,node)

	if not isLeaf:
	    nChildren = len(node)	
	    if isinstance(node, parsely.tree.StructNode):
		tagList = node.getType().tags
	    else:
		tagList = [None] * nChildren
	    for i in range(0,nChildren):
		child = node[i]
		childName = tagList[i]
		if childName is None:
		    childName = "[%s]" % i

		self.addNodeInfoToCTree(child, childName, n)

    def onSelect(self, tree, row, col, event):
	
	Tnode = tree.node_nth(row)
	Nnode = tree.node_get_row_data(Tnode)
	self.selected_row = Tnode
	if not Nnode or not Nnode.isLeaf():
	    self.valEnt.set_editable(0)
	    self.valEnt.set_text('')
	    self.spaceEnt.set_editable(0)
	    self.spaceEnt.set_text('')
	else:
	    self.valEnt.set_editable(1)
	    self.valEnt.set_text(Nnode.val)
	    self.spaceEnt.set_editable(1)
	    self.spaceEnt.set_text(Nnode.trailingSpace)
	self.selected_row = Tnode
	if self.dumpArea:
	    l = self.dumpArea.get_length()
	    if l != 0:
		self.dumpArea.delete_text(0,l)
	    self.dumpArea.set_position(0)
	    self.dumpArea.insert_defaults(str(Nnode))

    def onUnselect(self, tree,row,col,event):
	self.selected_row = None
	self.valEnt.set_editable(0)
	self.valEnt.set_text('')
	self.spaceEnt.set_editable(0)
	self.spaceEnt.set_text('')
	if self.dumpArea:
	    l = self.dumpArea.get_length()
	    if l != 0:
		self.dumpArea.delete_text(0,l-1)	    

    def onChanged(self, entry, which):
	tree = self.tree
	Tnode = self.selected_row
	if not Tnode: return
	Nnode = tree.node_get_row_data(Tnode)
	if not isinstance(Nnode, parsely.tree.LeafNode):
	    return
	if which == 'V':
	    self.tree.node_set_text(Tnode, 2, _q(entry.get_text()))
	    Nnode.set( entry.get_text() )
	else:
	    self.tree.node_set_text(Tnode, 3, _q(entry.get_text()))
	    Nnode.setSpace( entry.get_text() )

    def onTreeDestroy(self, tree):
	"destroy"
	del self.selected_row
	del self.valEnt
	del self.spaceEnt
	del self.tree


def onWinDestroy(w, state):
    gtk.mainquit()

def onDump(button, node):
    print ">>>"
    print node,
    print "<<<"

if __name__ == '__main__':
    import sys
    import parsely.grammar
    from parsely._util import fileContents, _trimFileFromArgv

    _trimFileFromArgv('browseTree')    
    if len(sys.argv) != 2:
        print "Syntax: browseTree <grammar> <file>"
        sys.exit(1)
    
    grammarFile = sys.argv[0]
    parseFile = sys.argv[1]
    format = parsely.grammar.generateFileFormat(fileContents(grammarFile))
    if not format: sys.exit(1)
    node = format.parse(fileContents(parseFile))

    a = BrowsingArea(node,1,1)
    t = TypeBrowsingArea(format.getTypeSystem())
    n = gtk.GtkNotebook()
    n.append_page(a, gtk.GtkLabel("Node"))
    n.append_page(t, gtk.GtkLabel("Types"))
    w = gtk.GtkWindow()
    w.connect("destroy", onWinDestroy, None)
    w.add(n)
    w.show_all()

    gtk.mainloop()
