#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Module to dump parsely formats into the standard 'grammar' form.  This
   is mainly used for debugging, but might help somebody else out in some
   weird way."""

import StringIO
from parsely._util import _escNonPrinting
import string
import parsely.format

def dumpFormat(format, f=None, V=0):
    """Writes out a format <format> to a file <f>.  If <f> is not
       specified, returns the format as a string.

       If <V> is set, writes out some clarifying comments as well.
       """
    useStringIO = (f is None)

    if useStringIO:
        f = StringIO.StringIO()

    ## options
    if V:
	f.write("### Options\n")
    for k,v in _sorted(format.Options.items()):
	if v == 1:
	    f.write("option %s;\n" %k)
	elif v == 0:
	    pass
	elif v is not None:
	    f.write("option %s = %s;\n" % (_id(k),_str(v)) )
    f.write("\n");	

    ## Default space
    if V:
	f.write("### Default Space\n");
    # The default space can be none if we haven't processed the format yet.
    if format.DefaultSpace is not None: 
	f.write("default space %s;\n\n" % _str(format.DefaultSpace))

    ## states
    if V:
	f.write("### States\n");
    for stateName in _sorted(format.States.keys()):
        state = format.States[stateName]
	if state.exclusive:
	    f.write("exclusive ");
	if state.noSpace:
	    f.write("nospace ");
	if state.onStart:
	    f.write("start ");
	f.write("state %s;\n" % _id(state.name))
    f.write("\n");

    # patterns 
    if V:
	f.write("### Patterns\n");
    for patName in _sorted(format.Patterns.keys()):
        pat = format.Patterns[patName]
	f.write("pattern %s = %s;\n" % (_id(pat.name), _pat(pat.val)))
    f.write("\n");

    # lexemes
    if V:
	f.write("### Lexemes\n");
    for lexeme in format.lexemeList:
	if lexeme.isToken():
	    f.write("token %s = %s" % (_id(lexeme.shortName), 
				       _pat(lexeme.pattern)))
	else:
	    f.write("space %s" % _pat(lexeme.pattern) )

	if lexeme.inStates:
	    f.write(" in %s" % 
		    string.join( map(_id,lexeme.inStates), ","))
	if lexeme.enterState:
	    f.write(" to %s" % _id(lexeme.enterState))
	if lexeme.isToken() and lexeme.default:
	    f.write(" default %s" % _str(lexeme.default))
	if lexeme.actionName:
	    f.write(" action %s" % _id(lexeme.actionName))
	f.write(";\n")
    f.write("\n")
	
    ## startsym
    if V:
	f.write("### Start symbol\n")
    f.write("start %s;\n\n" % _id(format.StartSymbol))

    ## rules
    if V:
	f.write("### Rules\n")
    for rule in format.ruleList:
	name = _id(rule.name)
	f.write("%s = %s;\n" % (name, _rule(rule.val,len(name))))
    f.write("\n")

    # actions
    if V:
	f.write("### Actions\n")
    for name,m in _sorted(format.Actions.items()):
	for lang,act in _sorted(m.items()):
	    f.write("action %s in %s = %s;\n" % 
		    (_id(name), _id(lang), _act(lang,act)))

    if useStringIO:
        v = f.getvalue()
        f.close()
        return v
    
def _id(s):
    return s

def _str(s):
    assert s is not None
    s = string.replace(s, '"', '\\"')
    s = string.replace(s, '\n', '\\n')
    return '"%s"' % s
    
def _pat(p):
    if isinstance(p, parsely.format.RE):
        w = p.pat.write(0,0,0)
	r = 're/%s/' % string.replace(w, "/", "\\/")
    else:
	r = _str(p.val)

    # only if it isn't int
    if p.pat.I == 1:
	r = r + 'i'
    return r

def _rule(r, nameLen=5):
    if isinstance(r, parsely.format.ALT):
	lst = []
	nameLen = nameLen + 3 # " = "
	for n,v in r.alternatives.items():
	    if _isNumeric(n):
		lst.append(_rule(v))
	    else:
		lst.append("[%s] %s" % (_id(n), _rule(v)))
	return string.join(lst, " |\n" + ' ' * nameLen)
    elif isinstance(r, parsely.format.SEQ):
	lst = []
	for i in range(len(r.members)):
	    if r.tags[i]:
		lst.append("%s:%s" % (_id(r.members[i]), _id(r.tags[i])))
	    else:
		lst.append(_id(r.members[i]))
	return string.join(lst, " ")
    elif isinstance(r, parsely.format.OPT):
	return "%s?" % _id(r.base)
    elif isinstance(r, parsely.format.MULTI):
	if r.exclusive:
	    x = "exclusive "
	else:
	    x = ""
	if r.sep:
	    return "%s%s separated %s%s" % (x,_id(r.sep), _id(r.base), 
					 _mult(r.min))
	elif r.term:
	    return "%s%s terminated %s%s" % (x,_id(r.term), _id(r.base), 
					  _mult(r.min))
	else:
	    return "%s%s" % (_id(r.base), _mult(r.min))
    else:
	assert 'this line' is 'never reached'
    
def _act(l,v):
    return '"""%v"""'

def _mult(m):
    if m == 0: return '*' 
    if m == 1: return '+'
    return ' %s...' % m

def _isNumeric(s):
    try:
	string.atoi(s)
	return 1
    except ValueError, e:
	return 0

def _sorted(l):
    l.sort()
    return l

if __name__ == '__main__':
    import sys
    from parsely.grammar import generateFileFormat
    from  parsely._util import _trimFileFromArgv, fileContents

    _trimFileFromArgv('dumpFormat')

    if len(sys.argv) == 0:
	dumpGrammar()
        sys.exit(0)
    
    format = generateFileFormat(fileContents(sys.argv[0]),V=0)

    dumpFormat(format, sys.stdout, 1)
