#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Code to generate flex-based scanners."""

import os
#import re, string, os

import parsely # for ParselyException
import parsely.cTrans

class FlexScannerGenerator:
    def __init__(self, format):
	self.lexemes = format.lexemeList
	self.startState = format.StartState
	self.states = []
	self.dotall = format.Options["dotall"]
	self.iscasei = format.Options["nocase"]
	self.module_name = format.Options["module"]
	self.names = parsely.cTrans.CIdentifiers(self.module_name, None)

	for name,state in format.States.items():
	    self.states.append( (name, state.exclusive) )
	    self.patterns = format.Patterns
	
    def write(self):
	file = open(self.names.getLexFilename(), 'w')
	self.writeTo(file)
	file.close()

    def writeTo(self,file):
	self.writeStandardPrologue(file)
	self.writeStandardOptions(file)
	self.writeStates(file)
	self.writePatterns(file)
	self.writeErrorCase(file)
	file.write("\n%%\n\n")
	self.writeStateChangeCode(file)
	self.writeLexemes(file)        
	file.write("\n%%\n\n")	
	self.writeStandardEpilogue(file)

    def writeStandardPrologue(self,file):
	'''XXXX-C'''
	file.write("""
%%{
  #include <glib.h>
  #include <parsely.h>
  #include "%(hfile)"
  #include "%(parserHfile)"

  static PrNode *_last_token = NULL;
  static const char *_initial_space = NULL;
  static GString *_most_recent_space = NULL;
  static gboolean _state_changed = FALSE;
  static int _next_state;

  static void _finish();
  static void _finish_last_token();
  static GNode *_new_token(const char*, int, const PrNodeType*);
  static void _more_space(const char*);
%%}
	""" % { 'hfile' : self.names.getHeaderFilename(),
        	'parserHfile' : self.names.getYaccHFilename() } )

    def writeStandardOptions(self, file):
	# XXXX-C
        file.write("""
%option XXXX-C""")

    def writeStates(self, file):
	"""XXXX-C"""
	for stateName, exclusive in self.states:
	    if exclusive: 
		flag = 'x'
	    else:
		flag = 's'
	    file.write("%" + flag + " " + self.getStateName(stateName) + "\n")
	file.write("\n")
    
    def writeStateChangeCode(self, file):
	"""XXXX-C"""
	file.write("    if (_state_changed) BEGIN(_next_state);\n\n");

    def writePatterns(self, file, patterns):
	"""XXXX-C"""
	for pattern in self.patterns.values():
	    name = pattern.name
	    pat = pattern.val.pat.write(flex=1, forceI=1)
	    file.write("%s   %s\n" % (self.getPatName(name)), pat)
	    file.write("\n")

    def writeLexemes(self,file):
	"""XXXX-C"""
	for lexeme in self.lexemes:
	    pat = lexeme.pattern.pat.write(flex=1, forceI=1)
	    isTok = lexeme.isToken()
	    states = lexeme.getStates()	
	    if len(states) > 0:
		firstTime = 1
		file.write("<")
		for s in lexeme.getStates():
		    if not firstTime: file.write(",")
		    file.write(getStateName(s))
		    firstTime = 0
		file.write(">")
	    file.write(pat)
	    file.write("    {\n")
	    # XXXX-C
	    if isTok:
		tokName = self.getTokName(lexeme.distinctName)
		typeName = self.getTypeName(lexeme.distinctName)
		file.write("    _last_token = _new_token(%stext,%s);\n" %
			   (self.names.yy, typeName))
		file.write("    return %s;\n    }\n" % tokName)
	    else: # space
		file.write("    _more_space(%stext);\n    }\n" % self.names.yy)
    
    def writeStandardEpilogue(self,file):
	file.write("""
static void _finish()
{
        _finish_last_token();  
}

static void _finish_last_token()
{
        PrNodeData* data;

	if (_last_token) {
  	        data = (PrNodeData*) _last_token->data;
		data->trailing_space = _most_recent_space->str;
        } else {
	        _initial_space = _most_recent_space->str;
        }
        g_string_free(_most_recent_space,0);
        _most_recent_space = g_string_sized_new(4); 
}

static PrNode* _new_token(const char* val, const PrNodeType* type)
{
        PrNodeData* data;
        PrNode* tok;
	
	_finish_last_token();

	data = g_new(PrNodeData,1);  
	data->code = PR_LEAF;
	data->type = type;
	data->data.leafData.val = g_strdup(val);
	data->data.leafData.trailing_space = NULL;
	node = g_node_new(data);
	_last_token = node;
	return node; 
} 

static void _more_space(const char* space)
{ 
        if (! _most_recent_space)
	        _most_recent_space = g_string_new(space);
        else
		g_string_append(_most_recent_space, space); 
} 

""")

class CFileGenerator:
    def __init__(self, format):
        self.lexemes = format.lexemeList
        self.rules = format.ruleList
        self.tokens = format.tokenList
        self.module_name = format.Options["module"]
        self.names = parsely.cTrans.CIdentifiers(self.module_name, None)

class BisonParserGenerator:
    def __init__(self, format):
	self.lexemes = format.lexemeList
	self.tokens = format.tokenList
	self.module_name = format.Options["module"]
	self.names = parsely.cTrans.CIdentifiers(self.module_name, None)
	self.defaultSpace = format.DefaultSpace # ????
	self.rules = format.ruleList
    
    def writeToFile(self, file):
	self.writePrelude(file)
	self.writeUnion(file)
	self.writeTokens(file)
	self.writeTypes(file)
	self.writeGrammar(file)
	file.write("%%\n\n")
	self.writeEpilog(file)
    
