#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Functions to scan .ply grammar-description files."""

import parsely
from parsely import format
from parsely.format import GrammarError
from hak_spark import GenericScanner, GenericParser
from parsely._util import fileContents, _unEscStr, UniqueIDDomain

import string,re
from types import TupleType,ListType,StringType

def generateFileFormat(input,V=0):
    s = Scanner()
    p = Parser()
    t = s.tokenize(input)
    if V:
        print "Tokens:"
        for tok in t:
            print tok
    if len(t) == 0:
	print "Error: empty grammar."
	return None

    ffMembers = p.parse(t)
    if V:
        print "FF ->", ffMembers
    f = format.FileFormat()
    errors = 0
    for m in ffMembers:
        line = m.lineNumber
	try:
            m.register(f)
	except GrammarError, e:
            print "%s: %s" % (line, e)
            errors = errors + 1
    try:
        f.process()
    except GrammarError, e:
	print e
        errors = errors + 1
    if errors:
	raise GrammarError('Errors prevented successful generation of grammar')
    else:
        return f

class Token:
    def __init__(self, type, val, lineno):
        self.type = type
        self.val = val
        self.lineNumber = lineno

    def __cmp__(self,o):
        return cmp(self.type,o)

    def __str__(self):
        return self.type + ": " + str(self.val)


class Scanner(GenericScanner):
    """Scanner used to read in parsely description files.  It's not
       very pretty, since it needs to handle C code, python code,
       perl re's, and several different kinds of quoting."""
       
    def __init__(self):
        GenericScanner.__init__(self)

    # List of all the keywords
    keywords =  {
        'action' :1,
        'default' :1,
        'exclusive' :1,
        'in' :1,
        'multi' :1,
        'nospace' :1,
        'option' :1,
        'pattern' :1,
        'separated' :1,
        'space' :1,
        'start' :1,
        'state' :1,
	'terminated'  :1,
        'token' :1,
        'to' :1,
        }

    _macroRE = re.compile(
        r'defmacro \s+ ([A-Za-z_][A-Za-z0-9_]*) \s* \( ( [^\)]* ) \).*',
        re.X)    

    def incLine(self, n):
	"""Increase the current line number by 'n'."""
	if n > self.macroLines:
	    self.lineNumber = self.lineNumber + (n - self.macroLines)
	    self.macroLines = 0
	else:
	    self.macroLines = self.macroLines - n

    def t_DEFMACRO(self, s):
        # We can't use the parser here, since we need to finish lexing before
        # we parse, but we can't be done lexing until the macros are done...
        r'defmacro \s+ [A-Za-z_][A-Za-z0-9_]* \s* \( [^\)]* \) \s* : \s*\n'

	self.incLine(string.count(s, '\n'))

        m = Scanner._macroRE.match(s)

        name = m.group(1)
        args = m.group(2)
        body = self.getPythonBlock()[0]

        if Scanner.keywords.has_key(name):
            raise GrammarError("Can't redefine keyword %s" % name)
        
        fnCode = "def %s(%s):\n%s" % (name, args, body)
        exec fnCode in self.macro_scope
        self.macros[name] = 1

    def t_DEF(self, s):
        r'def \s+ [A-Za-z_][A-Za-z0-9_]* \s* \( [^\)]* \) \s* : \s*\n'
	self.incLine(string.count(s, '\n'))
        body = self.getPythonBlock()[0]
        fnCode = s + body
        exec fnCode in self.macro_scope

    def t_INLINESTRING(self, s):
        r' << (\w+) '
        kwd = s[2:]
        self.rv.append(Token('STR', ["Placeholder for " + kwd, 0],
                             self.lineNumber))
        # UnEsc?
        self.inlineStrStack.append( (kwd, len(self.rv)-1, self.lineNumber ) )

    _importRE1 = re.compile(
	r'import \s+ ([A-Za-z_][A-Za-z_0-9\.]*)',
	re.X)

    def t_IMPORT1(self, s):
	r'import \s+ ([A-Za-z_][A-Za-z_0-9\.]*)'

	m = Scanner._importRE1.match(s)
        modulename = m.group(1)
	exec s in self.macro_scope
	expr = modulename + ".PARSELY_MACROS" 
	try:
	    macros = eval(expr, self.macro_scope)
	    for macro in macros:
		expr = modulename + "." + macro
		fn = eval(expr, self.macro_scope)
		self.macros[macro] = fn
		self.macro_scope[macro] = fn
	except:
	    pass

    def t_NUM(self, s):
        r'\d+'
        self.rv.append(Token('NUM', string.atoi(s), self.lineNumber))

    def t_OP(self, s):
        r'[?+*=;(),\.\[\]:|]'
        self.rv.append(Token(s,s,self.lineNumber))

    def t_RE(self,s):
        r'( m[^\n\s\w_0-9] | re[^\n\s\w_0-9] | / | q[^\n\s\w_0-9] )'
        self.rv.append( self.scanRE(s) )

    def t_PYTHON_IMMED(self,s):
        r'python: \s*\n'
	self.incLine(string.count(s, '\n'))
        body = self.getPythonBlock()[0]
        fnCode = "if 1:\n" + body
        exec fnCode in self.macro_scope

    def t_PYACTIONBLOCK(self,s):
        r'in [\s\n]+ [Pp]ython \s*: \s*\n '
	self.incLine(string.count(s, '\n'))
	lineNumber = self.lineNumber 
        body = self.getPythonBlock()[0]
        self.rv.append(Token('PYACTION',body, lineNumber))

    def t_CACTIONBLOCK(self,s):
        r'in [\s\n]+ [cC] [\s\n]+ = \{'
	self.incLine(string.count(s, '\n'))
        body = getCBlock()[0]
        self.rv.append(Token('CACTION',body, self.lineNumber))

    def t_STR1(self, s):
        r''' \"\"\" (?:\\.|\n|[^\"\\]|\"[^\"]|\"\"[^\"])* \"\"\" [iI]? '''
        casei = 0
        if s[-1] in 'iI':
            casei = 1
            s = s[:-1]
        val = _unEscStr(s[3:-3])
        self.rv.append(Token('STR', (val, casei), self.lineNumber))
	self.incLine(string.count(val, '\n'))

    def t_STR2(self, s):
        r""" \'\'\' (?:\\.|\n|[^\'\\]|\'[^\']|\'\'[^\'])* \'\'\' [iI]? """
        casei = 0
        if s[-1] in 'iI':
            casei = 1
            s = s[:-1]
        val = _unEscStr(s[3:-3])
        self.rv.append(Token('STR', (val, casei), self.lineNumber))
	self.incLine(string.count(val, '\n'))

    def t_STR3(self, s):
        r'''( \"(?:[^\"\\\n]|\\.|\\\n)*\"[iI]? |
              \'(?:[^\'\\\n]|\\.|\\\n)*\'[iI]? )'''
        casei = 0
        if s[-1] in 'iI':
            casei = 1
            s = s[:-1]
        val = _unEscStr(s[1:-1])
        self.rv.append(Token('STR', (val, casei), self.lineNumber))
	self.incLine(string.count(val, '\n'))

    def t_Z_RSTR1(self, s):
	r'''( \"(?:[^\"\\\n]|\\.|\\\n)*\n |
              \'(?:[^\'\\\n]|\\.|\\\n)*\n )'''
        raise GrammarError("Runaway string starting on line %s" % 
			   self.lineNumber)

    def t_Z_RSTR2(self, s):
        r""" \'\'\' (?:\\.|\n|[^\'\\]|\'[^\']|\'\'[^\'])* """
        raise GrammarError("Runaway string starting on line %s" % 
			   self.lineNumber)

    def t_Z_RSTR3(self, s):
        r''' \"\"\" (?:\\.|\n|[^\"\\]|\"[^\"]|\"\"[^\"])* \"\"\  '''
        raise GrammarError("Runaway string starting on line %s" % 
			   self.lineNumber)

    def scanChunk(self, endChar, nestOpen, nestClose, quoteMarkers,
                  quoteOpen, quoteClose, pyClose=0, depth=0, kind="element"):
        """Scan a 'chunk' of text.  'endChar' will indicate an end
           of the block.  nestOpen and nestClose are matched, nesting
           parentheis-like characters.  See uses below for examples."""

        startl = self.lineNumber
        sr = self.s_rest
        inQuote = 0
        qClose = None
        i=0
        while i < len(sr):
            if inQuote:
                if sr[i:i+len(qClose)] == qClose:
                    inQuote = 0
                    i = i + len(qClose) -1
                elif sr[i] == '\\':
                    i = i + 1
		    if i == len(sr):
			break
            elif sr[i] in nestOpen:  depth = depth + 1
            elif sr[i] in nestClose: depth = depth - 1
            elif sr[i] in quoteMarkers:
                for qi in range(0,len(quoteOpen)):
                    q = quoteOpen[qi]
                    if sr[i:i+len(q)] == q:
                        qClose = quoteClose[qi]
                        i = i + len(q)-1
                        break
                inQuote = 1
            elif sr[i] == '\\':
                i=i+1
            elif sr[i] == endChar: depth = -1

            if pyClose and i == len(sr) or \
	       (sr[i] == '\n' and 
		(i+1 == len(sr) or 
		 sr[i+1] not in ' \t\n#')):
                depth=-1

            i = i + 1
            if depth < 0: break

        if depth >= 0 or inQuote:
            raise GrammarError("Runaway %s on line %s" % (kind, startl))
        text = sr[:i-1]
        self.s_rest = sr[i:]#????

	self.incLine(string.count(sr[:i], '\n'))

        return text

    def scanRE(self,s):        
        casei = 0
        extended = 0
        startLine = self.lineNumber
	openingQual = s[:-1]
        endChar = s[-1]
        if endChar == '(': endChar = ')'
        elif endChar == '[': endChar = ']'
        elif endChar == '{': endChar = '}'
        elif endChar == '<': endChar = '>'

        reText = self.scanChunk(endChar, '({', '})', '[',
                                ('[',),
                                (']',), kind="regular expression")
            
        isCasei = 0
        isExtended = 0
        sr = self.s_rest
        i = 0
        while i < len(sr) and sr[i] in 'XxIi':
            if sr[i] in 'Xx':
                isExtended = 1
            else:
                isCasei = 1
            i = i + 1

        self.s_rest = sr[i:]
	if openingQual in ("re", "m", ""):
	    return Token('RE', (reText, isCasei, isExtended), 
                         startLine)
	else: # open == "q"
	    return Token('STR', (reText, isCasei), startLine)

    def t_WORD(self, s):
        r'[A-Za-z_][A-Za-z_0-9]*'
        if Scanner.keywords.has_key(s):
            self.rv.append(Token(s, None, self.lineNumber))
        elif self.macros.has_key(s):
            self.run_macro(s)
        else:
            self.rv.append(Token('ID', s, self.lineNumber))
        
    def t_whitespace(self, s):
        r'[\ \t]+'
        pass

    def t_nl(self, s):
        r'\n'
	self.incLine(1)
	
	# Handle inline strings
        while len(self.inlineStrStack) > 0:
            kwd, tokIdx,startline = self.inlineStrStack[0]
            kwd = kwd + "\n"
            sr = self.s_rest 
            lines = []
            line = ""
	    while line != kwd:
		try:
		    newlinePos = string.index(sr,"\n")
		except ValueError, v:
		    raise GrammarError("Runaway inline string <<%s on line %s" 
				       % (kwd[:-1], startline))
                line = sr[:newlinePos+1]
                lines.append(line)
                sr = sr[newlinePos+1:]
		self.incLine(1)
            lines = lines[:-1]
            self.rv[tokIdx].val[0] = string.join(lines,'')
            self.inlineStrStack = self.inlineStrStack[1:]
            self.s_rest = sr

    def t_comment(self,s):
        r'\#.*'
        pass

    def t_default(self, s):
        r'.'
        raise GrammarError("Unrecognized character <%s> on line %s" % (s,
                           self.lineNumber))

    def getPythonBlock(self):
        startLine = self.lineNumber
        block = self.scanChunk(None, '([{', '}])', '"\'#',
                              ('"', "'", '"""', "'''", '#'),
                              ('"', "'", '"""', "'''", '\n'),
                              1, kind="Python block")
	return (block, startLine)

    def getMacroArgs(self):
	sr = self.s_rest
        # Skip to right after the first (.
        i=0
        while i < len(sr):
            if sr[i] in ' \t\n':
                i = i + 1
                pass
            elif sr[i] == '(':
                i = i + 1
                break
            else:
                raise GrammarError("Expected arguments for macro")

	if i == len(sr):
	    raise GrammarError("Expected arguments for macro")

        self.s_rest = sr[i:]

        block = self.scanChunk(None, '([{', '}])', '"\'#',
                              ('"', "'", '"""', "'''", '#'),
                              ('"', "'", '"""', "'''", '\n'), 0, 0,
			      kind="macro arguments")

	return "("+ block + ")"

    def getCBlock(self):
        startLine = self.lineNumber
        # BUG- this allows the use of \ to escape an end-of-comment!
        block = self.scanChunk('}', '({[', '}])', '"\'/',
                               ('"', "'", '/*', '//'),
                               ('"', '"', '*/', '\n'),
			       kind="C block") 
	return (block, startLine)

    def run_macro(self, name):
	expr = name+self.getMacroArgs()
	# XXXX BUGBUG we don't beautify any exceptions here.
        result = eval(expr, self.macro_scope)
	if result is None:
	    return
	elif type(result) is StringType:
	    pass
	elif type(result) is ListType or type(result) is TupleType:
	    result = string.join(result)
	else:
	    raise GrammarError("Macro %s did not return a string." %name)

	self.s_rest = result + self.s_rest
	self.macroLines = self.macroLines + string.count(result,'\n')
        
    def tokenize(self, s):
        self.lineNumber = 1
	self.macroLines = 0
        self.rv = []
        self.inlineStrStack = []
        self.macros = {}
        self.macro_scope = {}
	self.macro_scope['MACROS'] = self.macros

	if parsely.use_pcre_hack:
	    code = self.re.code
	    while s:
		regs = code.match(s,0,len(s),re.ANCHORED)
		for i in range(1,len(regs)):
		    g = regs[i]
		    if (g[0] != -1 or g[1] != -1) \
		       and self.index2func.has_key(i-1):
			self.s_rest = s[g[1]:]
			self.index2func[i-1](s[:g[1]])
		s = self.s_rest
	else:
	    while s:
		match = self.re.match(s,0)
		assert match
		groups = match.groups()
		for i in range(len(groups)):
		    if groups[i] and self.index2func.has_key(i):
			self.s_rest = s[match.end():]
			self.index2func[i](groups[i])

		s = self.s_rest
		
        return self.rv
                        
class Parser(GenericParser):
    ## Fields:
    # strToSym: a StringToUID object.
    # multTab: a map from (multiplicity, name) pairs to (symbol,linenum)
    #    pairs.  This is not yet used!

    def __init__(self,start='File'):
        GenericParser.__init__(self, start)
	self.strToSym = StringToID()
	self.multTab = {}

    def multSym(mult,base,lineno):
	"""This is dead code that's never used.  Eventually, it will allow
           opt- and mult- nodes to appear embedded within SEQ nodes."""
	if mult == '+': mult = 1
	elif mult == '*': mult = 0
	if self.multTab.has_key((mult,base)):
	    sym,dline = self.multTab[mult,base][0]
	    if sym == None:
		raise GrammarError("Attempt to redeclare " +base + mult)
	if mult == '?':
	    sym = "Opt" + base
	elif mult == 1:
	    sym = "Multi" + base
	elif mult == 0:
	    sym = "OptMulti" + base
	else:
	    sym = base+str(mult)
	self.multTab[mult,base] = (sym,lineno)
	return sym

    def checkExplicitDecl(mult,base,lineno):
	"""This is dead code that's never used.  Eventually, it will allow
           opt- and mult- nodes to appear embedded within SEQ nodes."""
	if mult == '+': mult = 1
	elif mult == '*': mult = 0
	
	if self.multTab.has_key((mult,base)):
	    sym,dline = self.multTab[mult,base]
	    raise GrammarError(
		"Declaration on line %s conflicts with declaration of %s on line %s" % (lineno, dline,base))

	self.multTab[mult,base] = (None,lineno)	

    def p_File_1(self, args):
        ' File ::= Decl '
        if args[0]:
            return [ args[0] ]
        else:
            return []

    def p_File_2(self, args):
        ' File ::= File Decl'
        if args[1]:
            args[0].append(args[1])
        return args[0]

    def p_State_decl(self, args):
        ' Decl ::= StateModifiers state ID ; '
        s = parsely.format.State(stateName=args[2].val,
                                 start=args[0]['start'],
                                 noSpace=args[0]['nospace'],
                                 exclusive=args[0]['exclusive'])
        s.lineNumber = args[1].lineNumber
        return s    

    def p_StateModifiers_0(self, args):
        ' StateModifiers ::= '
        return {'start': 0, 'nospace':0, 'exclusive':0, 'lineno':-1}

    def p_StateModifiers_1(self, args):
        ''' StateModifiers ::= StateModifiers exclusive
            StateModifiers ::= StateModifiers start 
            StateModifiers ::= StateModifiers nospace '''
        args[0][args[1].type] = 1
        return args[0]
    
    def p_Pattern_decl(self, args):
        ' Decl ::= pattern ID = PatSpec ; '
        p = parsely.format.Pattern(args[1].val, args[3])
        p.lineNumber = args[0].lineNumber
        return p


    def p_Token_decl(self, args):
        ' Decl ::= token ID = PatSpec OptInSpec OptToSpec OptDefSpec OptAction ; '
	sym = self.strToSym.get(args[1].val)
        t = parsely.format.Token(name=sym,
                                 patspec=args[3],
                                 inStates=args[4],
                                 toState=args[5],
                                 default=args[6],
                                 action=args[7])
        t.lineNumber = args[0].lineNumber
        return t

    def p_OptEqPatSpec(self, args):
	''' OptEqPatSpec ::= 
	    OptEqPatSpec ::= = PatSpec '''
        if len(args) > 0:
	    return args[1]
	else:
	    return None

    def p_Token_decl_2(self, args):
	''' Decl ::= token STR OptEqPatSpec OptInSpec OptToSpec
	             OptDefSpec OptAction ; '''
        a = 2
	sym = self.strToSym.get(args[1].val[0])
	pat = args[2]
	if pat is None:
	    pat = parsely.format.LIT(args[1].val[0],
				     I=args[1].val[1])
	
        t = parsely.format.Token(name=sym,
                                 patspec=pat,
                                 inStates=args[3],
                                 toState=args[4],
                                 default=args[5],
                                 action=args[6])
        t.lineNumber = args[0].lineNumber
        return t
	
    def p_Symbol(self, args):
	'''Symbol ::= ID'''
        return self.strToSym.get(args[0].val)

    def p_Symbol_2(self, args):
	'''Symbol ::= STR'''
        return self.strToSym.get(args[0].val[0])

    
    def p_OptInSpec_0(self,args):
        ' OptInSpec ::= '
        return ()

    def p_OptInSpec_1(self,args):
        ' OptInSpec ::= in IdList '
        return args[1]

    def p_OptInSpec_2(self,args):
        ' OptInSpec ::= in * '
        return '*'

    def p_OptToSpec_0(self,args):
        ' OptToSpec ::= '
        return None

    def p_OptToSpec_1(self,args):
        ' OptToSpec ::= to ID'
        return args[1].val

    def p_OptDefSpec_0(self, args):
        ''' OptDefSpec ::= 
            OptDefSpec ::= default STR'''
        if len(args) > 0:
            return args[1].val[0]
        else:
            return None

    def p_OptAction(self, args):
        ''' OptAction ::= action ID
            OptAction ::=
            '''
        if len(args) > 0:
            return args[1].val
        else:
            return None

    def p_PatSpec_1(self,args):
        ' PatSpec ::= STR '
        return parsely.format.LIT(args[0].val[0],
                                  I=args[0].val[1])

    def p_PatSpec_2(self,args):
        ' PatSpec ::= RE '
        return parsely.format.RE(args[0].val[0],
                                 I=args[0].val[1],
                                 X=args[0].val[2])

    def p_Null_decl(self,ars):
        ' Decl ::= ; '
        return None

    def p_Space_decl(self, args):
        ' Decl ::= space PatSpec OptInSpec OptToSpec OptAction ; '
        s = parsely.format.Space(args[1], inStates=args[2],
                                 toState=args[3], action=args[4])
        s.lineNumber = args[0].lineNumber
        return s

    def p_Option_decl_1(self, args):
        ' Decl ::= option ID ; '
        o = parsely.format.Option(args[1].val)
        o.lineNumber = args[0].lineNumber
        return o

    def p_Option_decl_2(self, args):
        ' Decl ::= option ID = ID ;'
        o = parsely.format.Option()
        o.d[args[1].val] = args[3].val
        o.lineNumber = args[0].lineNumber
        return o

    def p_Option_decl_3(self, args):
        ' Decl ::= option ID = STR ;'
        o = parsely.format.Option()
        o.d[args[1].val] = args[2].val[0]
        o.lineNumber = args[0].lineNumber
        return o

    def p_DefSpace_decl(self,args):
        ' Decl ::= default space STR ;'
        ds = parsely.format.DefaultSpace(args[2].val[0])
        ds.lineNumber = args[0].lineNumber
        return ds

    def p_Start_decl(self,args):
        ' Decl ::= start ID ; '
        st = parsely.format.Start(args[1].val)
        st.lineNumber = args[0].lineNumber
        return st

    def p_Action_decl(self, args):
        ''' Decl ::= action ID OptKind in ID = STR ; '''
	name = args[1].val
	kind = args[2]
	lang = args[4].val
	code = args[6].val[0]
        return parsely.format.Action(name,lang,code,
				     kind=kind,
				     line=args[0].lineNumber)

    def p_Action_decl_python(self,args):
	''' Decl ::= action ID OptKind PYACTION '''
	name = args[1].val
	kind = args[2]
	code = args[3].val
	return parsely.format.Action(name,"python",code,
				     kind=kind,
				     line=args[0].lineNumber)

    def p_Action_decl_c(self,args):
	''' Decl ::= action ID OptKind CACTION '''
	name = args[1].val
	kind = args[2]
	code = args[3].val
	return parsely.format.Action(name,"c",code,
				     kind=kind,
				     line=args[0].lineNumber)

    def p_OptKind(self,args):
	''' OptKind ::= 
	    OptKind ::= ( ID ) '''
        if len(args) > 0:
	    return args[1].val
	else:
	    return None

    def p_Rule_decl(self,args):
        ' Decl ::= ID = RuleSpec ; '
        rs = parsely.format.Rule(args[0].val, args[2])
        rs.lineNumber = args[0].lineNumber
        return rs

    def p_RuleSpec_1(self,args):
        ' RuleSpec ::= AltRuleSpec '
	if (len(args) == 1) and (args[0][0] is None):
	    # We have a single untagged sequence.
	    return args[0][1]

        dict = {}
        alts = []
        for p in args[0]:
            if p[0] is not None:
                dict[p[0].val] = p[1]
            else:
                alts.append(p[1])
        return apply(parsely.format.ALT, tuple(alts), dict)

    def p_RuleSpec_2(self,args):
        ' RuleSpec ::= BaseRuleSpec '
        return args[0]

    def p_AltRuleSpec_1(self, args):
        ' AltRuleSpec ::= TaggedAlternative '
        return [ args[0] ]

    def p_AltRuleSpec_2(self, args):
        ' AltRuleSpec ::= AltRuleSpec | TaggedAlternative '
        args[0].append(args[2])
        return args[0]
    
    def p_TaggedAlternative_1(self,args):
        ' TaggedAlternative ::= [ ID ] SeqSpec '
        return ( args[1], args[3] )

    def p_TaggedAlternative_2(self,args):
        ' TaggedAlternative ::= SeqSpec '
        return ( None, args[0] )

    def p_BaseRuleSpec_1(self, args):
        ' BaseRuleSpec ::= Symbol ? OptAction'
	#checkExplicitDecl('?', args[0], args[1].lineNumber)
        return parsely.format.OPT(args[0],action=args[2])

    def p_BaseRuleSpec_2(self, args):
        ' BaseRuleSpec ::= Symbol Multiplicity OptAction'
	#checkExplicitDecl(args[1], args[0], -1) 
        return parsely.format.MULTI(args[0], args[1], action=args[2])
    
    def p_BaseRuleSpec_3(self, args):
        ''' BaseRuleSpec ::= OptExclusive Symbol separated Symbol
 	                     OptMultiplicity OptAction'''
        min = args[4]
        if min == 0: 
	    min = 1
        return parsely.format.MULTI(args[3], min=min, sep=args[1],
				    exclusive=args[0],
				    action=args[5])
        
    def p_BaseRuleSpec_4(self, args):
        ''' BaseRuleSpec ::= OptExclusive Symbol terminated Symbol
	                     OptMultiplicity OptAction '''
	min = args[4]
	if min == 0:
	    min = 1
	return parsely.format.MULTI(args[3], min=min, term=args[1],
				    exclusive=args[0], action=args[5])
    
    def p_OptExclusive(self, args):
	''' OptExclusive ::=
      	    OptExclusive ::= exclusive '''
        return len(args) > 0
	    
    #def p_BaseRuleSpec_5(self, args):
    #    ' BaseRuleSpec ::= SeqSpec '
    #    return args[0]

    def p_Multiplicity_1(self,args):
        ' Multiplicity ::= * '
        return '*'

    def p_Multiplicity_2(self,args):
        ' Multiplicity ::= + '
        return '+'

    def p_Multiplicity_3(self,args):
        ' Multiplicity ::= NUM . . . '
        return args[0].val

    def p_OptMultiplicity_0(self,args):
        ' OptMultiplicity ::= '
        return 0

    def p_OptMultiplicity_1(self,args):
        ' OptMultiplicity ::= Multiplicity '
        return args[0]

    def p_SeqSpec(self,args):
        ' SeqSpec ::= SeqElements OptAction'
	return apply(parsely.format.SEQ, tuple(args[0]), {'action' : args[1]})

    def p_SeqElements_1(self, args):
        ' SeqElements ::= '
        return []

    def p_SeqElements_2(self, args):
        ' SeqElements ::= SeqElements Symbol '
        args[0].append(args[1])
        return args[0]

    def p_SeqElements_3(self, args):
        ' SeqElements ::= SeqElements Symbol : ID '
        args[0].append(args[1] + ":" + args[3].val)
        return args[0]

    def p_IdList_1(self,args):
        ' IdList ::= ID '
        return [ args[0].val ]

    def p_IdList_2(self,args):
        ' IdList ::= IdList , ID '
        args[0].append(args[2].val)
        return args[0]

    def error(self, token):
	if token.val is not None:
	    vstr = '"%s" ' % str(token.val)
	else:
	    vstr = ''
        raise GrammarError("Syntax error at or near %s token %son line %s"
                           % (token.type, vstr, token.lineNumber))

    def resolve(self, list):	
	#
	# Only pick an ERROR production as a last resort.  There aren't
	# any yet, but I'll add them as common errors surface.
	#

	# print "AMBIGUITY!", list
	for el in list:	
	    if el[:5] != 'ERROR':
		return el
	return list[0]
	    


class StringToID(UniqueIDDomain):
    def __init__(self):
	UniqueIDDomain.__init__(self)

    # Map from some of the characters which can't apper in C identifiers
    # to strings which can appear in identifiers.  The '_transform'
    # function uses this map to try to make its identifiers more readable.
    # Strictly speaking, we could just replace everything with underscore,
    # but that wouldn't be as legible.
    _specialCharMap = {  
	'!' : 'EXC',    '"' : 'DQT',    '#' : 'SHP',    '$' : 'DLR',
	'%' : 'PCT',    '&' : 'AMP',    "'" : 'SQT',    '(' : 'LPR',
	')' : 'RPR',    '*' : 'STR',    '+' : 'PLS',    ',' : 'CMA', 
	'-' : 'DSH',    '.' : 'DOT',    '/' : 'SLS',    ':' : 'CLN',
	';' : 'SMI',    '<' : 'LAN',    '=' : 'EQL',    '>' : 'RAN', 
	'?' : 'QN',     '@' : 'AT',     '[' : 'LBR',    '\\' : 'BSL', 
	']' : 'RBR',    '^' : 'CRT',    '`' : 'BQT',    '{' : 'LBR', 
	'|' : 'VBR',    '}' : 'RBR',    '~' : 'TLD' }

    _specialCharRE = re.compile(r'[^A-Za-z0-9]')

    def _transform(self, id):
	return StringToID._specialCharRE.sub(_replaceSingleChar, id)

def _replaceSingleChar(ch):
    return StringToID._specialCharMap.get(ch.group(0),'_')

def dumpGrammar():
    for name in dir(Parser):
	if name[0:2] == 'p_':
	    print getattr(Parser,name).__doc__

if __name__ == '__main__':
    import sys
    from parsely._util import _trimFileFromArgv
    _trimFileFromArgv('grammar')

    if len(sys.argv) == 0:
	dumpGrammar()
        sys.exit(0)
    
    format = generateFileFormat(fileContents(sys.argv[0]),V=0)
    if len(sys.argv) == 2:
        node = format.parse(fileContents(sys.argv[1]))
        print node.sExpression()
