#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Types to represent the parse tree of a parsed file, and methods to 
   manipulate them.

   A Parsely tree has three fundamental types of nodes: leaf nodes,
   struct nodes, and list nodes. 
   
   * LeafNode represent leaves of the tree.  They contain the string
     value of a token, and the space which appeared immediately
     following that token.  The space is used to reconstruct the
     original file.

   * StructNodes represent a sequence of tokens, as generated by
     a sequence production in the grammar.  For example, if the struct
     node S corresponds to a production of
           Name = Word:firstname Word:lastname;
     then the components of S may be accessed either as S[0] and S[1], or
     as S.firstname and S.lastname.

   * List nodes represent a list of tokens as generated by a MULTI
     production in the grammar.  Their components may be accessed as
     L[0], L[1], ...

   * Additionally, FileNodes represent the root of the tree.  The space
     preceding the first token in the file is associated with the 
     FileNode.

   Features:

   - Converting/coercing a node to a string (i.e., str(node)) will 
     yield the string value for that node and all its children, as they
     would be written to the file.  The trailing space is not included.

   - You may insert strings into a parse tree, with no ill effect.  They
     will be silently converted to LeafNodes.  If you don't want them to
     carry the default space, however, you should use the setSpace method
     afterwards.
   
   - Because structures may have optional members, components may have
     members equal to None.

   - BUG: Before you write a filenode, you must call the cleanSpace method,
     to ensure that some default space appears in all tokens.

   - For convenience, you can access nodes by path.  For example, instead of
          file.x[3].z = "Hello world"
     you can write:
          file.setPath("x.3.z", "Hello world")

     A path is a string of identifiers and integers, separated by
     periods.  Paths also support a rudimentry kind of wildcard: when
     you're calling the getPath method, a path can contain a star to
     match all the children of a given node.  For example,
          file.getPath("3.*.z", "Hello world")
     gives a list of every file[3][??].z, for any value of ??.

   This module also has a few tools that may aid in node manipulation.
   
   - buildIndex: searches a portion of the tree for nodes matching a 
     cert

   - NodeVisitor: this class provides an implementation of the Visitor
     pattern for tree nodes.  

   - newNode: a factory method for making new node objects.  In general,
     it's preferable to call TypeSystem.newNode, or its alias 
     FileFormat.newNode.

   Finally, the tree module contains type objects for tree nodes.  When
   creating a new tree node, you should specify the node's type, especially
   if you want to access its fields (in the case of a StructNode) or its
   separators (in the case of a ListNode) properly.

   You can obtain type objects via the getType method of the TypeSystem
   object, or its alias in parsely.format.FileFormat.
   """

import string
from types import StringType, IntType, TupleType, ListType
from parsely._util import _wantList, _extendList, _escNonPrinting, \
     _namelist, _flattenList, replaceFile
import pprint

##################
# Types
##################

# This global variable sets the depth to which default nodes are constructed.
# 3 seems to work ok.
DEPTH = 3

class NodeType:
    """A NodeType represents the type of a Parsely node.  Every type
       has a name.  Some types have separate full names, of the form
       name:variant.  For example, the production:
          Person = [Humble] ID:first ID:last |
                   [Titled] Title:title ID:first ID:last;
       gives rise to two types:  'Person:Humble' and 'Person:Titled'.

       A type object knows how to construct instances of itself, and
       how to construct a default instance of itself.

       This is an abstract class."""

    ##Fields:
    # longName: the full name of this type.
    # typeName: the short name of this type.
    # variantName: the variant component of the full name, or None

    def __init__(self, typeName, variantName):
	"""Initializer for abstract class.  Sets up type name."""
	assert self.__class__ is not NodeType
        self.typeName = typeName
        if variantName:
            self.longName = typeName + ":" + variantName
        else:
            self.longName = typeName
        self.variantName = variantName

    def getName(self):
	"""Returns the name of this type"""
        return self.typeName

    def getLongName(self):
	"""Returns the full name of this type."""
        return self.longName

    def isNamed(self, name):
	"""Returns true if <name> is either of the names of this type."""
        return (self.typeName == name) or (self.longName == name)

    def getDefaultNode(self):
	"""Returns a new Node of the appropriate type for this object.

	   This method may not be valid until _constructDefaultNode has been
	   called."""
        raise "Tried to call method on abstract class NodeType"

    def _constructDefaultNode(self, typeSystem,depth):
	"""Initializes fields in this object necessary before getDefaultNode
	   can be called."""
        pass

    def isLeafType(self):
	"""Returns true iff this is the type for a LeafNode."""
        return 0

    def __str__(self):
	"""Returns the full name of this type."""
        return self.longName

    def _makeNode(self, val, trailingSpace=None):
	"""Returns a new node corresponding to the value <val>, with
	   trailingSpace <trailingSpace.  This method is used by newNode,
	   and shouldn't be called directly."""
        raise "Tried to call method on abstract class NodeType"

    def _buildDefaultNode(self, typeSystem, depth):
        """Called by '_constructDefaultNode' in other classes.  Returns
           either a default node for this type, or None if depth<0.
           Recursively calls _constructDefaultNode."""
        if depth < 0: return None
        self._constructDefaultNode(typeSystem,depth-1)
        return self.getDefaultNode()

class LeafNodeType(NodeType):
    """The type of a LeafNode."""
    
    ##Fields:
    # default: the default string for a leaf of this type.
    # defaultSpace: the default trailing space

    def __init__(self, typeName, default, defaultSpace):
        NodeType.__init__(self,typeName,None)
        self.default = default
        self.defaultSpace = defaultSpace

    def getDefaultNode(self):
        return LeafNode(self, self.default, self.defaultSpace)

    def getDefaultString(self):
	"""Returns the default value of a node with this type."""
        return self.default

    def getDefaultSpace(self):
	"""Returns the default trailing space of a node with this
  	   type."""
        return self.defaultSpace
    
    def isLeafType(self):
        return 1

    def _makeNode(self, val, trailingSpace=None):
        assert type(val) is StringType
	if trailingSpace is None: 
	    trailingSpace = self.defaultSpace
        return LeafNode(self, val, trailingSpace)
    
class StructNodeType(NodeType):
    """The type of a StructNode."""
    
    ##Fields:
    # memberTypes: a list of the names of the member types for this struct.
    # tags: a list of the tags for the members.
    # defaultMembers: a list of default nodes for the members of this struct.
    #     Set by _constructDefaultNode.

    def __init__(self, typeName, variantName, memberTypes, tags):
        NodeType.__init__(self,typeName,variantName)
        self.memberTypes = memberTypes
        self.tags = tags
        self.defaultMembers = 0 # Placeholder

    def _constructDefaultNode(self,typeSystem,depth):
        defMembers = []
        for m in self.memberTypes:
            t = typeSystem.getType(m,allMatches=1)
            if type(t) is ListType:
                # Pick the first alternative.
                t=t[0]
            if isinstance(t,LeafNodeType):
                defMembers.append(t.getDefaultNode())
            elif isinstance(t,ListNodeType):
                defMembers.append(t.getDefaultNode())
            elif isinstance(t,StructNodeType):
                defMembers.append(t._buildDefaultNode(typeSystem,depth))
            else:
                defMembers.append(None)

        self.defaultMembers = defMembers

    def getLen(self):
	"""Returns the number of elements in this struct."""
        return len(self.tags)

    def getDefaultNode(self):
        return StructNode(self, self.defaultMembers)

    def _makeNode(self, val, trailingSpace=None):
        assert type(val) is ListType
        assert len(val) == len(self.tags)
        return StructNode(self, val)

class ListNodeType(NodeType):
    """The type of a ListNode."""

    ##Fields:
    # baseType: The name of the base element type.
    # termType: The name of the terminator type.
    # sepType: The name of the separator type.
    # defSep: Default value for the separator.

    def __init__(self, name, baseType, termType=None, sepType=None,
                 exclusive=0):
	assert (termType is None) or (sepType is None)
        NodeType.__init__(self,name,None)
        self.baseType = baseType
        self.termType = termType
        self.sepType = sepType
        self.exclusive = exclusive
	self.defSep = None

    def getDefaultNode(self):
        return ListNode(self, [])

    def _constructDefaultNode(self,typeSystem,depth):
        if self.sepType:
            self.defSep = typeSystem.getType(self.sepType) \
                          ._buildDefaultNode(typeSystem,depth)
        elif self.termType:
            self.defSep = typeSystem.getType(self.termType) \
                          ._buildDefaultNode(typeSystem,depth)
            
    def getDefaultSeparator(self):
	return self.defSep.clone()
                
    def _makeNode(self, val, trailingSpace=None):
        assert type(val) is ListType
        if self.exclusive:
            return ExListNode(self,val)
        else:
            return ListNode(self, val)

    def getDefaltSeparator(self):
        """Returns a node for the default separator or terminator for this
           type."""
        if self.defSep:
            return self.defSep.clone()
        else:
            return None

class TypeSystem:
    """A TypeSystem is a container and factory for all the types used in
       a particular file format.  You can use FileFormat.getTypeSystem()
       to obtain the TypeSystem."""

    ##Building:
    # To construct a TypeSystem, FileFormat and friends pass type
    # objects to the _addType method.  Once done, they must call _process.
    ##Fields:
    # typesByTypeName: A map from type name to a list of all types having that
    #    name.
    # typesByLongName: A map from full type name to the unique type having
    #    that name.
    # defaultSpace: The default space in this file format.

    def __init__(self, defaultSpace):    
        """Constructs a new,empty TypeSystem with the provided
           defaultSpace."""
        self.typesByTypeName = {}
        self.typesByLongName = {}
        self.defaultSpace = defaultSpace

    def getDefaultSpace(self):
	"""Returns the default space for this type system."""
	return self.defaultSpace

    def _addType(self, type):
	"""Inserts a new type into this type system."""
        n = type.getName()
        ln = type.getLongName()
        
        if self.typesByTypeName.has_key(n):
            self.typesByTypeName[n].append(type)
        else:
            self.typesByTypeName[n] = [ type ]

	# Format should enforce this assertion.
	assert isinstance(type, LeafNodeType) \
	       or not self.typesByLongName.has_key(ln)
        self.typesByLongName[ln] = type

    def getType(self, name, nMembers=-1, allMatches=0):
	"""Returns the best match for a type with a given name.
	   If we're given a short name, and multiple resolutions are
	   possible, prints a warning message.

           If allMatches is true, returns a list of all matching types."""
        v = self.typesByLongName.get(name,None)
        if v: return v
        v = self.typesByTypeName.get(name,None)
	if v and len(v) == 1:
	    return v[0]
        elif v and len(v) > 1 :
            # If we know how many members will go into a StructNode, we can
            # use that information to guess the node's type.
            if nMembers > -1:
                matches = 0
                match = None
                for t in v:
                    if isinstance(t, StructNodeType):
                        if t.getLen()==nMembers:
                            matches = matches + 1
                            match = t
                    else:
                        matches = matches + 1
                        match = t
                if matches == 1:
                    return match
		elif matches > 1:
		    print "Warning: guessing type for %s" % name
		    return match
            if not allMatches:
                print "Warning: guessing type for %s" % name
                return v[0]
            else:
                return v[:]
        return None

    def _process(self):
	"""Called by format.  Must be called before getDefaultNode is
	   called on any type."""
        for el in self.typesByTypeName.values():
            for t in el:
                t._constructDefaultNode(self,DEPTH)

    def newNode(self, kind=None, val=None, trailingSpace=None):
	"""Create a new Node object.  

           kind -- a NodeType, or the name of a NodeType.
	   val -- a sequence, Node, a string, or None.
	   trailingSpace -- a string (for LeafNode).

	   If kind is not provided, but val is:
	       If val is a string: we construct a LeafNode.
	       If val is a sequence: we construct a ListNode.
	   If val is not provided, but kind is:
	       We construct a default node of type <kind>.

	   If val is a node:
	       We clone it."""
        if isinstance(kind,StringType):
            lng = 1
            if type(val) in (ListType, TupleType):
                lng = len(val)
            nodetype = self.getType(kind, lng)
            if nodetype is None:
                raise TypeErrror("No type named %s" % kind)
        else:
            nodetype = kind

        if isinstance(val, Node):
            return val.clone()
	elif type(val) is ListType:
	    val = map(self._valToNode, val)
	elif type(val) is TupleType:
            val = map(self._valToNode, list(val))

        if trailingSpace is None:
            trailingSpace = self.defaultSpace

        if nodetype and val is not None:
            return nodetype._makeNode(val, trailingSpace)
        elif nodetype:
            return nodetype.getDefaultNode()
        elif type(val) is StringType:
            return LeafNode(None,val,trailingSpace)
        elif type(val) in (ListType, TupleType):
	    return ListNode(None,val)
        else:
            raise TypeError("I can't make a node from %s" % val)

    def _valToNode(self, val):
        if val is None or isinstance(val,Node):
            return val
        elif type(val) is StringType:
            return LeafNode(None, val, self.defaultSpace)
        else:
            raise TypeError("Got '%s'; expected String or Node"
                            % `val`)


def newXNode(type=None, val=None, typesys=None, trailingSpace=None):
    """Creates a new node of a given type.

       This function is usually invoked as typesys.newNode(typename, val).
       This variant allows the creation of a node when no typesystem is
       present.

       It is strongly deprecated."""

    if typesys:
        return typesys.newNode(type,val,trailingSpace)

    if isinstance(val,Node):
	return val.clone()

    if isinstance(type,NodeType) and val is not None:
        return type._makeNode(val,trailingSpace)
    elif isinstance(type,NodeType):
        return type.getDefaultNode()
    
    if isinstance(val,StringType):
	return LeafNode(None,val,trailingSpace)
    elif isinstance(val,TupleType):
	return ListNode(None,list(val))
    elif isinstance(val,ListType):
	return ListNode(None,val[:])
    else:
	raise TypeError("I have no idea what kind of value " + val + " is.")

##################
# Nodes
##################
class Node:
    """Base class for all tree nodes."""

    def __init__(self):
        assert self.__class__ is not Node

    def clone(self):
        """Performs a deep copy of the tree under this node."""
        raise "Tried to call method on abstract class Node"

    def prettyPrint(self,V=0):
	"""Return a beautified s-expression for this node."""
	p = pprint.PrettyPrinter()
	p.pformat(self.sExpression(V))

    def sExpression(self,V=0):
        """Returns a nested tuple representing the value of this
           tree.  V ranges from 0 to 3, indicates the verbosity of
           the tuple."""
        raise "Tried to call method on abstract class Node"

    def dump(self,withTrailingSpace=1):
        """Returns the string encoded by this tree.  If <trailingSpace>
           is true, includes the space after the last token."""
        raise "Tried to call method on abstract class Node"

    def cleanSpace(self,defaultSpace):
	"""Fills in all 'unknown' space positions in this tree with the
	   default space value."""
        raise "Tried to call method on abstract class Node"

    def getPath(self,path):
	"""Obtains the descendent of this node at <path>, or raises
	   an exception if none exits.
	   
	   If <path> contains a star, returns a list of all the descendants
	   of this node matching the <path>."""

	if type(path) is StringType:
	    path = self._parsePath(path)
	return self._getPath(path)
	
    def setPath(self,path,val):
	"""Sets the descendent of this node at <path> to <val>, or raises
	   an IndexException if no corresponding location exists."""

	if type(path) is StringType:
	    path = self._parsePath(path)
	self._setPath(path,val)

    def _parsePath(self,path):
	"""Splits a path string into a list of ints and strings."""
	# We could have this build a function instead, but "Premature
	# optimization is the root of all evil." -- Knuth.
	m = []
	for el in string.split(path,'.'):
	    try:
		m.append(string.atoi(el))
	    except:
		m.append(el)
        return m

    def getType(self):
	"""Returns the NodeType of this node."""
        raise "Tried to call method on abstract class Node"

    def traverse(self, fn):
	"""Applies the function fn to this node, and to every node which
	   descends from it."""
        raise "Tried to call method on abstract class Node"

    def getTrailingSpace(self):
        """Returns the last space that could be found on any child of this
           node, or None."""
        raise "Tried to call method on abstract class Node"

    def allSubNodes(self):
	"""Returns a list of every node which descends from this one."""
	x = _allSubNodesBuilder()
	self.traverse(x)
	return x.members

    def isInclude(self):
        """Returns true if and only if this node is an include node."""
        return 0

    def isLeaf(self):
        """Returns true iff this node is a leaf."""
        return 0

class FileNode(Node):
    """Represents the root of a file.  All operations on a FileNode dispatch
       to the toplevel node of a file."""
    ##Fields:
    # _val - the root node for the file.
    # _initialSpace - a string holding the space at the beginning of
    #    the file
    # _defaultSpace - the default space for all subnodes in the file.
    #    We use this for the cleanSpace method.
    # _fileName - the name of the file we read to generate this
    #    file.
    
    def __init__(self, val, initialSpace='', defaultSpace='@@', fileName=None):
        """Creates a new FileNode around the value <val>, with
           defaultSpace, initialSpace, and fileName as specified."""
	self.__dict__['_val'] = val
	self.__dict__['_initialSpace'] = initialSpace
        self.__dict__['_defaultSpace'] = defaultSpace
        self.__dict__['_fileName'] = fileName

    def getFileName(self):
	"""Returns the file name for this file"""
	return self._fileName

    def getInitialSpace(self):
        """Returns the space at the beginning of this file"""
        return self._initialSpace

    def setInitialSpace(self, val):
        """Sets the space at the beginning of this file"""
        self.__dict__['_initialSpace'] = val

    def cleanSpace(self,ds=None):
	if ds is None:
	    ds = self._defaultSpace
        self._val.cleanSpace(ds)

    def clone(self):
	newVal = self._val
	if newVal: 
	    newVal = newVal.clone()
	return FileNode(newVal, self._initialSpace, self._defaultSpace)

    def sExpression(self,V=0):
        if V == 2 and self._val:
            return ( 'FILE [%s]' % self._initialSpace,
                     self._val.sExpression(V))
	if self._val:
	    return ( 'FILE', self._val.sExpression(V) )
	else:
	    return ( 'FILE', None )

    def __str__(self):
        return self.dump()

    def __repr__(self):
        return 'FileNode(initialSpace=%s, defaultSpace=%s, val=%s)' % (
            `self._initialSpace`, `self._defaultSpace`, `self._val`)

    def getType(self):
        return self._val.getType()

    def dump(self,withTrailingSpace=1):
        self.cleanSpace()
        return self._val.dump(withTrailingSpace)

    def traverse(self,fn):
        fn(self)
        self._val.traverse(fn)

    def setVal(self,val):
        """Sets the node containing this file's data"""
        self.__dict__['_val'] = val

    def getVal(self):
        """Returns the node containing this file's data"""
        return self._val
    
    def __getattr__(self,name):
        return getattr(self._val, name)

    def __setattr__(self,name, val):
        setattr(self._val, name, val)

    def dumpAllFilesToDict(self):
	"""Returns a dictionary mapping file names to file contents, for every
	   FileNode under this node."""
        dict={}
	def dumpFilesToDictTraversalFunc(node,dict=dict):
	    if isinstance(node,FileNode):
		dict[node.getFileName()] = node.dump()
	self.traverse(dumpFilesToDictTraversalFunc)
	return dict

    def flushOne(self, fileName=None):
	"""Replaces the file for this FileNode with the modified version."""
	if fileName is None:
	    fileName = self.getFileName()
	if fileName is None:
	    raise ParselyException("Tried to dump file with an unknown name.")

	self.cleanSpace()
	contents = self.dump()
	replaceFile(fileName, contents)

    def flush(self):
	self.flushOne()

    #def flushAll(self):
    #	# XXXX This will not work if we include files we don't own,
    #	# XXXX but never change those files.
    #	"""Replaces this file, and all files beneathe it, with modified
    #       versions."""
    #	self.cleanSpace()
    #    dict = self.dumpFilesToDict()
    #	for fname,contents in dict.items():
    #	    _replaceFile(fname,contents)

class IncludeNode(Node):
    """An IncludeNode indicates that an include operation has occurred
       for a given node.  It has two parts: the included part, and the
       include directive itself.  By default, it behaves as if it were
       the included part, but the getInclude() method yields the include
       directive.

       BUG: Includes are disabled.  They just don't work.

       BUG: There is not at present any way for a user to create an
       IncludeNode, except during the scan/parse phase."""

    ##Fields:
    # _included
    # _directive
    # _type
 
    def __init__(self, included, directive):
        assert included.__class__ == FileNode
        self.__dict__['_included'] = included
        self.__dict__['_directive'] = directive
        if included:
            self._type = included.getType()

    def isInclude(self):
        return 1

    def getInclude(self):
        return self._directive

    def setInclude(self,val):
        self._directive = val

    ##For Node.
    def clone(self):
        return IncludeNode(self._included.clone(), self._directive.clone())

    def sExpression(self,V=0):
        return ('INCLUDE',
                self._directive.sExpression(V),
                self._included.sExpression(V))
    
    def dump(self,withTrailingSpace=1):
        return self._directive.dump(withTrailingSpace)

    def cleanSpace(self,defaultSpace):
        self._directive.cleanSpace(defaultSpace)
        self._include.cleanSpace(defaultSpace)

    def traverse(self,fn):
        fn(self)
        self._directive.traverse(fn)
        self._include.traverse(fn)

    def __repr__(self):
        return 'IncludeNode(directive=%s, included=%s)' % (
            self._directive, self._included)

    def __getattr__(self,name):
        return getattr(self._val, name)

    def __setattr__(self,name, val):
        setattr(self._val, name, val)
        
    
class LeafNode(Node):
    """A LeafNode  holds a single token along with its trailing space."""
    ##Fields:
    # val: A string containing the value of this leaf.
    # trailingSpace: A string containing the trailing space after this leaf.
    #    May be None, for 'unknown.'
    # _type: The LeafNodeType for this leaf, or None.
    # lineNumber: the starting line for this token, or -1 for unknown.
    
    def __init__(self, type, val, trailingSpace, line=-1):
	self._type = type
        if (val == None or val == ()) and type:
            val = type.getDefaultString()
        if (trailingSpace == None) and type:
            trailingSpace = type.getDefaultSpace()
	self.val = val
	self.lineNumber = line
	self.trailingSpace = trailingSpace

    def isLeaf(self):
        return 1

    def getType(self):
	return self._type

    def clone(self):
        return LeafNode(self._type, self.val, self.trailingSpace)

    def __repr__(self):
	return "LeafNode(%s,%s)" % (repr(self.val), 
                                    repr(self.trailingSpace))

    def __str__(self):
	return self.val

    def __cmp__(self, s):
        return cmp(self.val,s)
        
    def getTrailingSpace(self):
        return self.trailingSpace

    def getSpace(self):
        """Returns the space immediately following this token."""
        return self.trailingSpace

    def setSpace(self,val):
        """Sets the space immediately following this token."""
        self.trailingSpace = val

    def cleanSpace(self, ds):
        if self.trailingSpace is None:
            if self._type is not None:
                self.trailingSpace = self._type.defaultSpace
            else:
                self.trailingSpace = ds

    def get(self):
        """Returns the value of this leaf."""
        return self.val

    def set(self,val):
        """Replaces the value of this leaf."""
        self.val = val

    def sExpression(self,V=0):
	typename = None
	if self._type: typename = self._type.longName
        if V == 3:
            return "(%s %s [%s])" % (typename,
                                     _escNonPrinting(self.val),
                                     self.trailingSpace)
        elif V == 2:
            return (typename, _escNonPrinting(self.val))
        elif V == 1:
            return _escNonPrinting(self.val)
        else:
            return self.val

    def dump(self, withTrailingSpace=1):
        if withTrailingSpace and (self.trailingSpace is not None):
            return self.val + self.trailingSpace
        else:
            return self.val

    def _getPath(self,path):
        if len(path) > 0:
            raise IndexException(
                "Couldn't get path '" + string.join(path,'.') + 
                "' on leaf node.")
        else:
            return self

    def _setPath(self,path,val):
	raise "Couldn't set path '" + string.join(path,'.') + "' on leaf node."

    def traverse(self,fn):
	fn(self)

    def getLineNumber(self):
        return self.lineNumber

class _BranchNode(Node):
    """_BranchNode is an abstract base class common to all nodes with multiple
       children -- namely, ListNode and StructNode."""

    ##Fields:
    # _members: A list of the children of this node.
    # _type: The NodeType for this node, or None.
    
    def __init__(self, type_, val):
        """Initializes a new _BranchNode with the given type and value.

           Requires: val is a list of Node and None."""
        assert self.__class__ is not _BranchNode
	assert type(val) is ListType
	self.__dict__['_type'] = type_
        
        self.__dict__['_members'] = val[:]

    def getType(self):
	return self._type

    def _valToMember(self,val):
        """If <val> is not a valid Node, try to make it one, or raise
           a TypeError.  Returns the resulting node."""
        if val is None or isinstance(val,Node):
            return val
        elif type(val) is StringType:
            return LeafNode(None, val, self.getTrailingSpace())
        else:
            raise TypeError("Got '%s'; expected String or Node"
                            % `val`)

    def _valsToMembers(self,val):
        """Given a list of values, applies _valToMember to them all."""
	if val is None:
	    return ()
	elif type(val) in (TupleType, ListType):
	    return map(self._valToMember, val)
        elif type(val) is StringType or isinstance(val, Node):
            return [ self._valToMember(val) ]
	else:
            raise TypeError("Expected a list of values")

    def clone(self):
        assert "this" is "not called"

    def cleanSpace(self,ds):
        for m in self._members:
            if m is not None:
                m.cleanSpace(ds)

    def __len__(self):
        return len(self._members)

    def __coerce__(self, o):
        if type(o) is StringType:
            return (str(self), o)
        
    def sExpression(self,V=0):
        tstr = "????"
        if type:
            tstr = self._type.longName
        mstrs = []
        for m in self._members:
	    if m:
		mstrs.append(m.sExpression(V))
	    else:
		mstrs.append(m)
        return (tstr,) + tuple(mstrs)

    def __str__(self):
        return self.dump(0)

    def dump(self, withTrailingSpace=1):
        if withTrailingSpace:
            return string.join(map(dumpNode, self._members),'')
        elif len(self._members) > 0:
            return (string.join(map(dumpNode, self._members[:-1]), '')
                    + dumpNode(self._members[-1],0))
        else:
            return ''

    def _setMember(self, idx, val):
	"""Common code for setIdx and setTag: ensures that
	   'val' is a valid node, or creates a node to hold it."""
        if type(val) == StringType:
            if isinstance(self._members[idx], LeafNode):
                self._members[idx].val = val
            else:
                if self._members[idx] is None:
                    space = self.getTrailingSpace()
                else:
                    space = self._members[idx].getTrailingSpace()
                self._members[idx]  = LeafNode(None, val, space)
	else:
	    self._members[idx] = self._valToMember(val)

    def __getitem__(self, idx):
        """Tries to return the <idx>th child of this node."""
        return self._members[self._rangeIdx(idx)]

    def __setitem__(self, idx, val):
        """Tries to set the <idx>th child of this node."""
        self._setMember(self._rangeIdx(idx), val)

    def _rangeIdx(self, idx):
        """Helper function: raises errors if idx is bad.  Returns idx
           on sucess."""
        if type(idx) is not IntType:
            raise TypeError("Index not an integer")
        elif 0 <= idx < len(self._members):
            return idx
        else:
            raise IndexError("Index out of bounds")
        
    def __nonzero__(self):
	return 1

    def getTrailingSpace(self):
        for i in range(len(self._members)-1,-1,-1):
            if self._members[i] is not None:
                s = self._members[i].getTrailingSpace()
                if s is not None:
                    return s
        return None

    def _getPath(self,path):
	if len(path) == 0:
            return self

        if type(path[0]) == IntType:
            return self[path[0]]._getPath(path[1:])
        elif path[0] == '*':
            elements = []
            for m in self._members:
                if not m: continue
                try:
                    res = m._getPath(path[1:])
                    if type(res) in (TupleType, ListType):
                        _extendList(elements,res)
                    else:
                        elements.append(res)
                except:
                    pass
            return elements
        else:
            return getattr(self,path[0])._getPath(path[1:])
		
    def _setPath(self,path,val):
	if len(path) == 1:
	    if type(path[0]) == IntType:
		self[path[0]] = val
	    else:
		setattr(self,path[0],val)
	else:
	    if type(path[0]) == IntType:
		self[path[0]]._setPath(path[1:],val)
	    else:
		getattr(self,path[0])._setPath(path[1:],val)

    def traverse(self,fn):
	fn(self)		
	for m in self._members:
	    if m:
		m.traverse(fn)

    def getLineNumber(self):
        for m in self._members:
            if not m: continue
            l = m.getLineNumber()
            if l > 0: return l
        return -1

class StructNode(_BranchNode):
    """A StructNode represets a sequence of tokens, as generated by a sequence
       production in the grammar."""
    def __init__(self, type, val):
        """Initializes this StructNode with the given type and value.
        
           Requires: val is a list of Node and None."""
        if type and val == None:            
            val = [None] * type.getLen()
	_BranchNode.__init__(self,type,val)

    def clone(self):
        newM = []
        for m in self._members:
            newM.append(m.clone())
	return StructNode(self._type, newM)

    def __getTagIdx(self,tag):
        """Returns the index corresponding to <tag>, or raises an attribute
           error if it couldn't find it."""
	if not self._type: 
	    raise AttribueError("Node has unknown type, so I couldn't find %s"
				% tag)
        for idx in range(0,len(self._type.tags)):
            if self._type.tags[idx] == tag:
                return idx
        raise AttributeError(tag)

    def __setattr__(self, name, val):
        """Sets the child with tag <name> to equal <val>."""
        self._setMember(self.__getTagIdx(name),val)

    def __getattr__(self, name):
        """Returns the child with tag <name>."""
        return self._members[self.__getTagIdx(name)]

    def __repr__(self):
        return 'StructNode(%s)' % (repr(self._members))
    
class ListNode(_BranchNode):
    """A ListNode is a series of elements, as generated by a list production
       in the grammar.
       
       Requires: val is a list of Node and None."""
    
    def __init__(self, type, val):
	_BranchNode.__init__(self,type,val)

    def clone(self):
        newM = []
        for m in self._members:
            newM.append(m.clone())
        return ListNode(self._type, newM)
    
    def append(self,val):
        """Adds a new element to the end of the list."""
        self._members.append(self._valToMember(val))

    def extend(self,lst):
        """Adds a list of elements to the end of the list"""
        if isinstance(lst, ExListNode):
            lst = _alternatingElements(lst._members)
        elif isinstance(lst, _BranchNode):
            lst = lst._members
        _extendList(self._members,self._valsToMembers(lst))

    def insert(self, idx, val):
        """Inserts <val> at position <idx> of this list"""
	self._members.insert(idx,self._valToMember(val))

    def __delitem__(self, idx):
        """Removes an item from the list."""
	del self._members[idx]

    def __getslice__(self, i, j):
        """Returns a slice from the list."""
	return self._members[i:j]

    def __setslice__(self, i,j, s):
        """Replaces a slice from the list."""
	self._members[i:j] = self._valsToMembers(s)
    
    def __delslice__(self, i,j):
        """Removes a slice from thie list."""
	del self._members[i:j]

    def __repr__(self):
        return 'ListNode(%s)' % (repr(self._members))

    def __setattr__(self, attr, val):
        raise AttributeError(attr)

class ExListNode(ListNode):
    """An ExListNode is a ListNode with for an exclusive list (one where
       separators/terminators don't count."""

    ##Fields:
    # isSep: boolean: Is this list separated? (As opposed to terminated)

    def __init__(self, type, val):
        assert type.sepType or type.termType
        self.__dict__['isSep'] = type.sepType is not None
        ListNode.__init__(self,type,val)

    def clone(self):
        newM = []
        for m in self._members:
            newM.append(m.clone())
        return ExListNode(self._type, newM)

    def append(self,val):
        self.extend(self,(val,))

    def extend(self,lst):
        if isinstance(lst, ExListNode):
            self.extend(_alternatingElement(lst._members))
            return
        elif isinstance(lst,_BranchNode):
            lst = lst._members
            
        if self.isSep:
            self._members.append(self.getDefaultSeparator())
	_extendList(self._members, self._fillWithDefaultSeparator(lst))
        self._trimLast()

    def insert(self,idx,val):
        self[idx:idx] = (val,)
    
    def __delitem__(self,idx):
        del self._members[idx+idx]
        if idx+idx < len(self._members):
            del self._members[idx+idx]
	self._trimLast()

    def __len__(self):
        return (len(self._members)+1) / 2

    def _rangeIdx(self, idx):
        """Helper function: raises errors if idx is bad.  Returns idx
           on sucess."""
        if type(idx) is not IntType:
            raise TypeError("Index not an integer")
        idx = idx + idx
        if 0 <= idx < len(self._members):
            return idx
        else:
            raise IndexError("Index out of bounds")

    def __getslice__(self,i,j):
	i = i + i
	if j < 2000000000: # MAXINT
	    j = j + j
        return _alternatingElements(self._members[i:j])

    def __setslice__(self,i,j,s):
	i = i + i
	if j < 2000000000: # MAXINT
	    j = j + j
        self._members[i:j] = self._fillWithDefaultSeparator(s)
        self._trimLast()

    def __delslice__(self,i,j):
	i = i + i
	if j < 2000000000: # MAXINT
	    j = j + j
        del self._members[i:j]
	self._trimLast()

    def getSeparator(self,idx):
        return self._members[idx+idx+1]

    def setSeparator(self,idx,val):
        self._members[idx+idx+1] = self._valToMember(val)

    def getDefaultSeparator(self):
        """Returns a new node equivalent to the default separator for this
           list."""
        return self._type.getDefaultSeparator()

    def _fillWithDefaultSeparator(self,lst):
        """Given a list <l1, l2, l3> returns <l1 S l2 S l3 S> where S is
           the default separator."""
        lst = self._valsToMembers(lst)
        r = []
        for item in lst:
            r.append(item)
            r.append(self.getDefaultSeparator())
        return r

    def _trimLast(self):
        """Helper function to make sure the last element is correct."""
        if self.isSep:
            if (len(self._members) % 2) == 0:
                del self._members[-1:]
        else:
            if (len(self._members) % 2) == 1:
                self._members.append(self.getDefaultSeparator())

def dumpNode(node,withTrailingSpace=1):
    """Returns the string stored in a given node and its children."""
    if isinstance(node,StringType):
        return node
    elif node is None:
	return ""
    else:
        return node.dump(withTrailingSpace)

####
# Visualization
####

def sExpressionToString(sExpression):
    """Given an sExpression, returns a string equivalent."""
    if type(sExpression) is StringType:
        return sExpression
    if type(sExpression) is NoneType:
        return "NONE"
    return "(" + string.join(map(sExpressionToString,sExpression), ' ') + ")"

###
# Helper class to build closure
###

class _allSubNodesBuilder:
    """Helper class used in allSubNodes traversal."""
    def __init__(self):
	self.members = []

    def __call__(self,node):
	self.members.append(node)

def buildIndex(node, pathToNodes=None, indexPath=None, typeSet=None, unique=0,
	       descend=0, debug=0):
    """Constructs a map from strings to nodes.

       The nodes will be chosen from among node.getPath(pathToNodes), or
       just <node> if the path is unspecified.  If <descend> is true, then
       children of these nodes will be considered as well.

       Only nodes the names of whose types are in the list typeSet will be
       considered.

       If <indexPath> is set, then the string mapping to a given node
       'n' will be found at n.getPath(indexPath).  (If no matching
       node is found, 'n' is not included.)  Otherwise, str(n) is used
       as the index for 'n'.

       If <unique> is set, then every string maps to the last matching
       node with that index.  Otherwise, each string maps to a list of
       all matching nodes with that index."""

    # Parse the paths.
    if pathToNodes:
	pathToNodes=node._parsePath(pathToNodes)
    if indexPath:
	indexPath=node._parsePath(indexPath)

    # Construct typeSet.
    if type(typeSet) in (TupleType, ListType):
	ts = {}
	for el in typeSet:
	    ts[el] = 1
	typeSet = ts

    # Build the list 'consideredNodes' to contain only the nodes
    # with match the getPath, descend, and typeSet constraints.
    if not pathToNodes:
	consideredNodes = node 
    else:
	consideredNodes = node._getPath(pathToNodes)

    if type(consideredNodes) is not ListType:
	consideredNodes = [ consideredNodes ]

    if not consideredNodes: return {}

    if descend:
	consideredNodes = _flattenList(map(Node.allSubNodes, consideredNodes))

    if typeSet is not None:
	def isValidType(node,ts=ts):
            if node is None or type(Node) is StringType: return 0
            nodetype = node.getType()
            return nodetype and (ts.has_key(nodetype.getLongName()) or
                                 ts.has_key(nodetype.getName()))
	consideredNodes = filter(isValidType,consideredNodes)

    # For every considered node, perhaps add it to the index.
    index = {}
    for N in consideredNodes:
	ival = None
        if debug: print 'Considering node >>', N, '<<'
        # _getPath might fail, so wrap this with an exception.
	try:
	    if indexPath is  None:
		ival = N.dump(0)
	    else:
		ival = N._getPath(indexPath).dump(0)
            
            if debug: print 'Index val >>', ival, '<<'

	    if unique:
		index[ival] = N
	    elif index.has_key(ival):
		index[ival].append(N)
	    else:
		index[ival] = [N]
	except:
	    pass

    return index

###
# Visitor
###

class NodeVisitor:
    """NodeVisitor is a useful base class for complex traversal operations.

       To use NodeVisitor, define methods of the form
           'visit_NAME(self,node)',
       where NAME is either a short type name, a long type name with
       the colon replaced by '__', or one of 'ListNode, StructNode,
       LeafNode, Default'.

       When you call the 'visit' method of a NodeVisitor with some
       particular node, it will invoke the most appropriate of the visit_
       methods.

       Use the visitChildren method to recursively visit all the children
       of a given node."""

    ##Fields:
    # methods: a map from type name to method.

    def __init__(self):
        assert(self.__class__ is not NodeVisitor)
        self.methods = { 'ListNode' : None,
                         'StructNode': None,
                         'LeafNode' : None,
                         'Default' : None }

        for n in _namelist(self):
            if n[:5] != 'visit':
                continue
            rest = n[5:]
            if len(rest) == 0 or rest == '_children':
                continue
            self.methods[rest] = getattr(self,n)            
            rest2 = string.replace(rest,'__',':')
            self.methods[rest2] = getattr(self,n)

    def visit(self, node):
        if node is None: return None
        type = node.getType()        
        if type is not None:
            m = self.methods.get(type.longName,None)
            if m is None:
                m = self.methods.get(type.typeName, None)
            if m is None:
                cl = node.__class__
                while cl in (FileNode, IncludeNode):
                    if cl is FileNode:
                        cl = node._val.__class__
                    else:
                        cl = node._included.__class__
                if cl is LeafNode:
                    m = self.methods['LeafNode']
                elif cl is ListNode or cl is ExListNode:
                    m = self.methods['ListNode']
                elif cl is StructNode:
                    m = self.methods['StructNode']
            if m is None:
                m = self.methods['Default']          
            if m is None:
                return None
            return m(node)
        
    def visitChildren(self,node,replace):
        """Visit all the children of a given node.  If <replace> is
           true, use the return values of the visit methods as
           the new values of <node>'s children."""
        if isinstance(node,_BranchNode) or isinstance(node,FileNode):
            i = 0
            while i < len(node):
                res = self.visit(node[i])
                if replace:
                    node[i]=res
                i = i + 1

    def visit_Default(self, node):
        self.visitChildren(node,0)

def _alternatingElements(lst):
    """Given a sequence of a0, a1, a2, a3, a4.... returns a list of
       a0, a2, a4, a6..."""
    r = []
    l = len(lst)
    for i in range(0,l,2):
        r.append(lst[i])
        i = i + 2
    return r

