#!/usr/bin/python

# Parsely - A cross-language tool for parsing and file manipulation.
#
# Copyright (C) 1999-2000 Nick Mathewson
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

import types, string, os
import parsely
from parsely import remanip

######################################################################
## REGULAR EXPRESSIONS
######################################################################

def reManipTest(Group, p, I=0, L=0, dump=None, written=None, nogroup=None,
		casei=None, iscasei=None, default=None, resolveWith=None,
                perlism=None, V=0):

    def show(which,wanted,got,V=V, G=Group):
        G.t(which, got, wanted)
        
    pat = remanip.ParsedRE(p,I=I,L=L)
    if resolveWith:
        for k in resolveWith.keys():
            if isinstance(resolveWith[k],types.StringType):
                resolveWith[k] = remanip.ParsedRE(resolveWith[k])
        pat = pat.resolve(remanip.resolverFromMap(resolveWith))
    if dump != None:
	show("remanip:str", dump, str(pat))
    if written != None:
        show("remanip:written", written, pat.write())
    if nogroup != None:
        show("remanip:written(nogroup)", nogroup, pat.write(noGrouping=1))
    if casei != None:
        show("remanip:toCaseI", casei, pat.toCaseI().write())
    if iscasei != None:
        show("remanip:isCaseI", iscasei, pat.I)
    if default != None:
        show("remanip:default", default, pat.makeDefault())
    if perlism != None:
        show("remanip:perlism", perlism, pat.isPcreOnly())
   
def testReManip(V=0):
    "The remanip module"
    patterns = [
	# Simple characters.
	(r'AB x3', {"dump":    'Group(C(A),C(B),C( ),C(x),C(3))',
		    "written": r'AB x3',
                    "nogroup": r'AB x3',
		    "casei":   r'[Aa][Bb] [Xx]3',
		    "iscasei": 0,
		    "default": r'AB x3',
                    "perlism": 0} ),
	# Alternations
	(r'34|45', {"dump":    'Group(Alt(Group(C(3),C(4)),Group(C(4),C(5))))',
		    "written": r'34|45',
                    "nogroup": r'34|45',
		    "casei":   r'34|45',
		    "iscasei": 2,
		    "default": '34',
                    "perlism" : 0} ),
	# Groups and multiplicities
	(r'(@@)+:{3,4}',
	           {"dump":    'Group(Multi(Group(C(@),C(@)),1..None)' +
		               ',Multi(C(:),3..4))',
		    "written": r'(@@)+:{3,4}',
                    "nogroup": r'(?:@@)+:{3,4}',
		    "casei":   r'(@@)+:{3,4}',
		    "iscasei": 2,
		    "default": '@@:::',
                    "perlism": 0 } ),
	(r'(1|2)* {,2}',
	           {"dump":    'Group(Multi(Group(Alt(Group(C(1)),' +
		               'Group(C(2)))),0..None),' +
		               'Multi(C( ),0..2))',
		    "written": '(1|2)* {0,2}',
                    "nogroup": '(?:1|2)* {0,2}',
		    "casei":   '(1|2)* {0,2}',
		    "iscasei": 2,
		    "default": '',
                    "perlism": 0 } ),
	(r'a{5,}',
	           {"written": r'a{5,}',
		    "casei":   r'[Aa]{5,}',
		    "iscasei": 0,
		    "default": 'aaaaa' } ),
        (r'a|b?',
                   {"dump": 'Group(Alt(Group(C(a)),Group(Multi(C(b),0..1))))',
                    "written": r'a|b?',
		    "casei":   r'[Aa]|[Bb]?',
		    "iscasei": 0,
		    "default": 'a',
                    "perlism": 0 } ),         
        # Nongreedy matches
        (r'\s+?x{2,}?',
	           {"dump":    'Group(Multi(CType(s),1..None?),'+
                                     'Multi(C(x),2..None?))',
		    "written": r'\s+?x{2,}?',
		    "casei":   r'\s+?[Xx]{2,}?',
		    "iscasei": 0,
		    "default": ' xx',
                    "perlism": 'non-greedy pattern matching' } ),         
	# Character types and classes.
	(r'\s\d\w\S\D\W', 
	           {"dump":    'Group(CType(s),CType(d),CType(w),' + 
		               'CType(S),CType(D),CType(W))',
		    "written": r'\s\d\w\S\D\W',
		    "casei":   r'\s\d\w\S\D\W',
		    "iscasei": 2,
		    "default": ' 0aA  ',
                    "perlism": 0 } ),
	(r'[abc]+[^abc]+',
	           {"written": r'[abc]+[^abc]+',
		    "casei":   r'[A-Ca-c]+[^A-Ca-c]+',
		    "iscasei": 0,
		    "default": 'aA',
                    "perlism": 0 } ),
	(r'[\d\w][^\d\w]',
	           {"written": r'[\d\w][^\d\w]',
		    "casei":   r'[\d\w][^\d\w]',
		    "iscasei": 2,
		    "default": 'A!' } ),
        (r'[A-Za-z0-9@]',
                   {"dump": 'Group(CClass(0,Chars(C(A)..C(Z))' + 
                            'Chars(C(a)..C(z))Chars(C(0)..C(9))C(@)))',
                    "written": r'[A-Za-z0-9@]',
                    "casei":   r'[A-Za-z0-9@]',
                    "iscasei": 2,
                    "default": 'A' } ),
	# Character escapes
	(r'\n\t[\000\010]\x60',
	           {"written": r'\n\t[\000\010]\x60',
		    "casei":   r'\n\t[\000\010]\x60',
		    "iscasei": 2,
		    "default": '\n\t\010`' } ),
	(r'\$\+\*\{\}\(\)\ \"',
	           {"written": r'\$\+\*\{\}\(\)\ \"',
		    "casei":   r'\$\+\*\{\}\(\)\ \"',
		    "iscasei": 2,
		    "default": '$+*{}() "',
                    "perlism": 0 } ),
	(r'\\',
	           {"written": r'\\',
		    "casei":   r'\\',
		    "iscasei": 2,
		    "default": '\\' } ),
	# Special characters.
	(r'^.+$',
	           {"dump": "Group(Special(^),Multi(Special(.),1..None)," +
		            "Special($))",
		    "written": r'^.+$',
		    "casei":   r'^.+$',
		    "iscasei": 2,
		    "default": 'A',
                    "perlism": 0 } ),
        (r'a\b',
                   {"dump": "Group(C(a),Special(\\b))",
                    "written": r'a\b',
                    "iscasei" : 0,
                    "default": 'a',
                    "perlism": 'word-boundary' } ),
	# Special groups
	(r'(?:a)',
	           {"dump": "Group(Group(C(a)))",
		    "written": '(a)',
                    "nogroup": '(?:a)',
		    "casei":   r'([Aa])',
		    "iscasei": 0,
		    "default": 'a',
                    "perlism": 0 } ),
	(r'(?!a).',
	           {"dump": "Group(Group(C(a)),Special(.))",
		    "written": '(?!a).',
                    "nogroup": '(?!a).',
		    "casei":   r'(?![Aa]).',
		    "iscasei": 0,
		    "default": 'A',
                    "perlism": 'zero-width assertion' } ),
        (r'(?#a).',
	           {"dump": "Group(Lit(),Special(.))",
		    "written": r'.',
		    "casei":   r'.',
		    "iscasei": 2,
		    "default": 'A',
                    "perlism": 0 } ),
        (r'(?ix)y',
	           {"dump": "Group(Group(C(x)),C(y))",
		    "written": '(?ix)y',
                    "nogroup": '(?ix)y',
		    "casei":   r'(?ix)[Yy]',
		    "iscasei": 0,
		    "default": 'xy',
                    "perlism": 0 } ),
	# Intrinsic literals
	(r'Pars(el|le)y',
	           {"L": 1,
		    "dump" :   r'Lit(Pars\(el\|le\)y)',
		    "written": r'Pars\(el\|le\)y',
		    "casei":   r'[Pp][Aa][Rr][Ss]\([Ee][Ll]\|[Ll][Ee]\)[Yy]',
		    "iscasei": 0,
		    "default": 'Pars(el|le)y' } ),
	# Intrinsic casei
	(r'[A]a',
	           {"I": 1,
		    "written": r'[A]a',
		    "casei":   r'[Aa][Aa]',
		    "iscasei": 1,
		    "default": 'Aa' } ),
	# Patrefs
        (r'{:x:}+y',
                  {"dump": 'Group(Multi(Ref(x),1..None),C(y))',
                   "written": r'{:x:}+y',
                   "casei":   r'(?i{:x:})+[Yy]',
                   "iscasei": 0 } ),
        (r'{:X:}+y',
                  {"resolveWith": {'X': r'[ .]'},
                   "written": r'([ .])+y',
                   "nogroup": r'(?:[ .])+y',
                   "casei":   r'([ .])+[Yy]',
                   "iscasei": 0,
                   "perlism": 0 } ),
	]
    ok = 0
    bug = 0
    for p in patterns:
	G = _tGroup('The pattern /%s/' % p[0],2,V=V)
        apply(reManipTest,(G,p[0],), p[1])
        o,b = G.result()
        ok,bug = ok + o, bug+b
    G = _tGroup('CleanXRE',2,V=V)
    xpatterns = [
        ('a b \s [a b\ c]', r'ab\s[ab\040c]'),
        (r'''\s+ # Match a bunch of spaces
             \S  # Followed by a single non-space
             ''', r'\s+\S')
        ]
    for p,q in xpatterns:
        G.t('/%s/' % p, remanip.CleanXRE(p), q)
    o,b = G.result()
    ok,bug = ok + o, bug+b
    return (ok,bug)

# Error cases
#     Bogus regexen:  '[\d-3]'.  
#     Nonexistant references.
#     OTHERS

def testReErrors(V=0):
    """Regular expressions with errors"""
    patterns = [
	(r'(bad', 'Unterminated group' ),
	(r'\V', "Unrecognized alphanumeric escape '\\V'"),
	(r'\A', "Unrecognized alphanumeric escape '\\A'"),
	(r'\3', "Backreference '\\3' not supported"),
	(r'(?+foo)', 'Extension syntax for (?+...) not supported'),
	(r'[ ', 'Runaway character class'),
	(r'[\d-9]', 'Malformed range in character class'),
	(r'[1--9]', 'Malformed range in character class'),
	(r'[5-]', 'Unterminated range in character class'),
	(r'[-5]', 'Malformed range in character class'),
	(r'\xZZ', "Bad hex escape starting with '\xZZ'"),
	(r'{:9023:}', "Malformed pattern reference"),	
    ]
    G = _tGroup('Error messages', 1,V=V)
    for p,q in patterns:
	exception = None
	try:
	    pat = remanip.ParsedRE(p)
        except remanip.REError, msg:
	    exception = msg
	G.t('The invalid pattern /%s/' % p, str(exception), str(q))

    # Bad patref
    exception = None
    try:
	pat = remanip.ParsedRE(r'{:Word:}')
	pat = pat.resolve(remanip.resolverFromMap({}))
    except remanip.REError, msg:
	exception = msg
    G.t('Nonexistant pattern reference', str(exception), 
	'No declaration for pattern reference Word')

    return G.result()


######################################################################
## LEXICAL ANALYSIS
######################################################################

def lexTest(G, descr, scanner, tokenization):
    """tokenization of form "(space, (ttype, val, space)...)" """

    v = [tokenization[0]]
    for t in tokenization[1:]:
        v.append(t[1])
        v.append(t[2])
    s = string.join(v,'')
    res = scanner.tokenize(s)

    resTokenization = [ res[0] ]
    for t in res[1:]:
        t = t.val
        resTokenization.append( (t._type.typeName, t.val, t.trailingSpace))

    resTokenization = tuple(resTokenization)
    G.t(descr, resTokenization, tokenization)

def make_lexing_formats():
    from parsely.format import *
    lst1 = (
        Pattern("Word", RE(r"""[A-Za-z_] # An alphanumeric character or uscore
                               [A-Za-z0-9_]* # Followed by stuff.""",X=1)),
        Pattern("NiSym", LIT("NI",I=1)),
        Pattern("EndsInNi", RE(r"{:Word:}{:NiSym:}")),
        Pattern("StringChar", RE(r'[^"\\]|\\"|\\\\')),
        Space(RE(r"[ \t\n]+")),
        Space(RE(r"#.*$")),
        Token("Bob", LIT("Bob")),
        Token("Author", LIT("Nick")),
        Token("Author", LIT("Mathewson")),        
        Token("Eq", LIT("=")),
        Token("Number", RE(r'\d+')),
        Token("String", RE(r'"{:StringChar:}*"')),
        Token("NiSym", RE(r'{:EndsInNi:}')),
        Token("TheWordNiSym", LIT(r'{:NiSym:}', I=1)),
        Start("File"),        
        Rule("File", SEQ("Bob")))
    basic = FileFormat(lst1)
    basicI = FileFormat( lst1, Option("nocase"))
    stateful = FileFormat(
        State("EqImmed", noSpace = 1, exclusive=1),
        State("Command", exclusive = 0, start = 1),
        State("Val", exclusive = 0),
        State("CComment", exclusive=1),
        Space(RE(r'[ \t]')),
        Token("CMD", RE(r'[a-z]'), inStates="Command", toState="EqImmed"),
        Token("EQ", LIT("="), inStates="EqImmed", toState="Val"),
        Token("WORD", RE(r'([A-Za-z_]|\\ )+'), inStates=("Command", "Val"),
              toState="Val"),
        Token("SHARP", LIT("#"), inStates="*"),
        Token("NL", LIT("\n"), inStates=("Command", "EqImmed", "Val"),
              toState="Command"),
        Space(LIT("/*"), inStates=("Command"), toState="CComment"),
        Space(LIT("*/"), inStates="CComment", toState="Command"),
        Space(RE(r'.|\n'), inStates="CComment"),
        Start("File"),
        Rule("File", SEQ("CMD"))
        )
    stateSpace = FileFormat(
	State("L", exclusive=1,start=0,noSpace=1),
	State("I", exclusive=0,start=1),
	Space(RE(r'[ \t\n]+')),
	Space(RE(r'-+'), inStates=("L")),
	Token("Open", LIT("("), toState="L"),
	Token("Close", LIT(")"), inStates="L", toState="I"),
	Token("ID", RE("[A-Za-z_]+"), inStates="L"),
	Start("File"),
	Rule("File", MULTI("LST")),
	Rule("LST", SEQ("Open", "IDs", "Close")),
	Rule("IDs", MULTI("IDs", "+"))
	)

    basic.process()
    basicI.process() 
    stateful.process()
    stateSpace.process()
    return basic.scanner, basicI.scanner, stateful.scanner, stateSpace.scanner
        
def testPCRELex(V=0):
    """PCRE Lexical analysis"""
    basic_lex,basicI_lex,stateful_lex,statespace_lex = make_lexing_formats()
    S1 = ("#And a few\n#introductory comments ....\n   ",
          ("Bob", "Bob", "   # Is a pretty nifty guy\n  "),
          ("Bob", "Bob", ""),
          ("Eq", "=", " "),
          ("NiSym", "niftyNi", "\n"),
          ("Number", "6060843", "  # And I'm waiting for you!\n"),
          ("String", '"A string may contain \\"escapes\\" and \\\\s."', ""))

    S2 = ("#And a few\n#introductory comments ....\n   ",
          ("Bob", "bOb", "   # Is a case-insensitive guy\n  "),
          ("Bob", "BOB", ""),
          ("Eq", "=", " "),
          ("NiSym", "knightswhosayNi", "\n"),
          ("Number", "6060842", "  # And I'm waiting for you!\n"),
          ("Author", "Nick", " "),
          ("Author", "Mathewson", "\n"),
          ("String", '"A string may contain \\"escapes\\" and \\\\s."', ""))

    S3 = ("/* This is a test of\n * multi-line, c-style comments. */  ",
          ("CMD", "a", ""), ("EQ", "=", " "),
          ("WORD", "Foob\\ ar", ""), ("NL", "\n", " "),
          ("CMD", "z", ""), ("SHARP", "#", ""),
          ("EQ", "=", " "), ("WORD", "x", ""), ("NL", "\n", ""),
          ("WORD", "W", " "), ("SHARP", "#", ""),
          ("WORD", "W", " "), ("WORD", "W", " "))

    S4 = ("   ", ("Open", "(", "--"), ("ID", "A", "-"), 
	  ("ID", "B", ""), ("Close", ")", "\n\t") )

    G = _tGroup("Basic lexical analyzers", level=2,V=V)
    lexTest(G, "Basic scanner", basic_lex, S1)
    lexTest(G, "Basic casei scanner#1", basicI_lex, S1)
    lexTest(G, "Basic casei scanner#2", basicI_lex, S2)
    lexTest(G, "Stateful scanner", stateful_lex, S3)
    lexTest(G, "Scanner with state-specific space", statespace_lex, S4)

    # Extra space here - should render invalid.
    S5 = ("", ("CMD", "a", " "), ("EQ", "=", " "),
          ("WORD", "Foob\\ ar", ""), ("NL", "\n", " "),
          ("CMD", "z", ""), ("SHARP", "#", ""),
          ("EQ", "=", " "), ("WORD", "x", ""), ("NL", "\n", ""),
          ("WORD", "W", " "), ("SHARP", "#", ""),
          ("WORD", "W", " "), ("WORD", "W", " "))
    try:
	lexTest(G, "Should fail", stateful_lex, S5)
	G.t('Nospace',0)
    except parsely.pcre_scanner.ScanError, msg:
	G.t('Nospace', str(msg),
          '1 characters of unrecognized input on line 1 [pos=1 state=EqImmed]')

    return G.result()

def testNoLexingHack(V=0):
    if not parsely.use_pcre_hack:
	return (0,0)

    parsely.use_pcre_hack = 0

    basic_lex = make_lexing_formats()[0]
    S1 = ("#And a few\n#introductory comments ....\n   ",
          ("Bob", "Bob", "   # Is a pretty nifty guy\n  "),
          ("Bob", "Bob", ""),
          ("Eq", "=", " "),
          ("NiSym", "niftyNi", "\n"),
          ("Number", "6060843", "  # And I'm waiting for you!\n"),
          ("String", '"A string may contain \\"escapes\\" and \\\\s."', ""))
    G = _tGroup("Lexical analysis without pcre_hack",level=2,V=V)

    lexTest(G, "Basic scanner", basic_lex, S1)

    parsely.use_pcre_hack = 1
    return G.result()

######################################################################
## PARSING 
######################################################################

def parseTest(G, descr, format, input, sExpression):

    result = format.parse(input).sExpression()

    G.t(descr, result, sExpression)

def make_simple_grammar():
    from parsely.format import *
    ff = FileFormat(
        Space(RE('[ \t]+')),
        Token("EOL", RE(r'\n+')),
        Token("LBR", LIT("[")),
        Token("RBR", LIT("]")),
        Token("EQ",  LIT("=")),
        Token("FROB", LIT("Frob",I=1)),
        Token("PATH", LIT("PATH",I=1)),
        Token("CMD", LIT("CMD",I=1)),
        Token("COLON", LIT(":")),
        Token("COLON", LIT("CLN")), # Synonym for COLON.
        Token("SEMI", LIT(";")),
        Token("X1", LIT("x1",I=1)),
        Token("X2", LIT("x2",I=1)),
        Token("WORD", RE("\w+")),
        Start("File"),
        Rule("OptEol", OPT("EOL")),
        Rule("File", MULTI("Section",min=1)),
        Rule("Section", SEQ("OptEol", "Header:head", "Body:body")),
        Rule("Header", SEQ("LBR", "Word:k", "RBR", "EOL")),
        Rule("Body", MULTI("Decl")),
        Rule("Decl", ALT(decl=SEQ("OptFrob:frob", "WORD:key", "EQ", "Val:val",
                                  "EOL"),
                         pathdecl=SEQ("PATH", "EQ", "PathSpec:ps", "EOL"),
			 cmddecl=SEQ("CMD:cmd", "EQ", "CmdSpec:cs", "EOL"),
			 x1decl=SEQ("X1", "X1Spec:x1s", "EOL"),
			 x2decl=SEQ("X2", "X2Spec:x2s", "EOL"),
			 )),
	Rule("Word", ALT(w="WORD", p="PATH", f="FROB")),
        Rule("OptFrob", OPT("FROB")),
        Rule("Val", MULTI("Word",min=1)),
        Rule("PathSpec", MULTI("Word", min=1, sep="COLON")),
	Rule("CmdSpec", MULTI("Word", min=1, term="SEMI")),
	Rule("X1Spec", MULTI("Word", min=1, sep="FROB", exclusive=1)),       
	Rule("X2Spec", MULTI("Word", min=1, term="SEMI", exclusive=1)),
        )
    ff.process()
    return ff

def testParserAndNodes(V=0):
    """Earley parser; simple node manipulation"""

    ex1 = """

    [sec1]

    k=v a z
    frob w = s
    z=path frob

    [sec2]

    [sec3  ]
    path = x:y CLN z
    cmd = x;y;
    """

    p1 = ("FILE",
          ("File",
           ("Section",
            "\n\n",
            ("Header", "[", ("Word:w", "sec1"), "]", "\n\n"),
            ("Body",
             ("Decl:decl", None,
              "k", "=", ("Val", ("Word:w", "v"),
                         ("Word:w", "a"),
                         ("Word:w", "z")), "\n"),
             ("Decl:decl", "frob",
              "w", "=", ("Val", ("Word:w", "s")), "\n"),
             ("Decl:decl", None,
              "z", "=", ("Val", ("Word:p", "path"), ("Word:f", "frob")),
              "\n\n"))),
           ("Section", None,
            ("Header", "[", ("Word:w", "sec2"), "]", "\n\n"),
            ("Body", )),
           ("Section", None,
            ("Header", "[", ("Word:w", "sec3"), "]", "\n"),
            ("Body",
             ("Decl:pathdecl", "path", "=",
              ("PathSpec", ("Word:w", "x"), ":",
               ("Word:w", "y"), "CLN",
               ("Word:w", "z")), "\n"),
	     ("Decl:cmddecl", "cmd", "=",
	      ("CmdSpec", ("Word:w", "x"), ";",
	       ("Word:w", "y"), ";"), "\n")))
           )
          )

    kvl = make_simple_grammar()       
    G = _tGroup("Parsing and basic node manipulation", level=2,V=V)
    parseTest(G, "KVL example", kvl, ex1, p1)

    conf = kvl.parse(ex1)

    # Test indexing
    sections = parsely.tree.buildIndex(conf, '*', 'head.k',
                                       unique=1, descend=0)
    G.t("Index: unique", sections['sec1'] is conf[0])
    paths = parsely.tree.buildIndex(conf, '*.body.*', '0',
                                    typeSet=['Decl:pathdecl'], unique=0)
    G.t("Index: typeSet", paths['path'][0] is conf[2].body[0])
    paths2 = parsely.tree.buildIndex(conf, '', '0',
                                     typeSet=['Decl:pathdecl'], unique=0,
                                     descend=1)
    G.t("Index: descend", paths2['path'][0] is conf[2].body[0])
    
    # Test node setting.    
    G.t("Value access #1", conf[1].head.k, "sec2")
    G.t("Value access #2", conf[0].body[1].key, "w")
    G.t("Value access #3", conf[0][2][1].frob, "frob")
    G.t("Path access #1", conf.getPath("0.body.1.key"), "w")
    G.t("Path access #2", conf.getPath("0.body.1").dump(0), "frob w = s\n")
    G.t("Path access #3", map(parsely.tree.dumpNode,
                              conf.getPath("0.head.*")),
        ['[', 'sec1', ']', '\n\n    '])
    G.t("Path access #4", map(parsely.tree.dumpNode,
                              conf.getPath("*.head.k")),
        ['sec1', 'sec2', 'sec3  '])
        
    conf[2].head.k = "SecThree"
    G.t("Value mod -- Replacing T in S", conf[2].head.k, "SecThree")
    conf[1].head = "[SecTwo]\n\n"
    G.t("Value mod -- Replacing N in S", conf[1].head, "[SecTwo]\n\n")
    conf[0].body[1] = "Y = P\n"
    G.t("Value mod -- Replacing N in M", conf[0].body[1], "Y = P\n")
    conf[0].body[2].val[0] = "Yop"
    G.t("Value mod -- Replacing T in M", conf[0].body[2].val[0], "Yop")
    conf.setPath("2.body.1.cs.0", "BAR")
    G.t("SetPath #1", conf[2].body[1].cs[0], "BAR")
    conf.setPath("2.body.1.cmd", "command")
    G.t("SetPath #2", conf[2].body[1].cmd, "command")
    
    G.t("Representation", `conf[0].head.k`,
        "StructNode([LeafNode('sec1','')])")

    # Also tests 'traverse'
    G.t("AllSubnodes", map(parsely.tree.dumpNode,
                           conf[0].head.allSubNodes()),
        ['[sec1]\n\n    ', '[', 'sec1', 'sec1', ']', '\n\n    '])

    exr = """

    [sec1]

    k=v a z
    Y = P
    z=Yop frob

    [SecTwo]

    [SecThree  ]
    path = x:y CLN z
    command = BAR;y;
    """
    G.t("Modifications preserve formatting", str(conf), exr)

    class fooVisit(parsely.tree.NodeVisitor):
        def __init__(self):
            self.LNodes = 0
            self.otherDecls = 0
            self.thePathDecl = None
            parsely.tree.NodeVisitor.__init__(self)

        def visitListNode(self,node):
            self.LNodes = self.LNodes+1
            self.visitChildren(node,0)
                                
        def visitDecl(self,node):
            self.otherDecls = self.otherDecls+1

        def visitDecl__pathdecl(self,node):
            self.thePathDecl = node

        def visitDefault(self,node):
            self.visitChildren(node,0)
    
    # Restore conf
    conf = kvl.parse(ex1)
    # Try visitor
    Visitor = fooVisit()
    Visitor.visit(conf)
    G.t("Visitor", (Visitor.otherDecls, Visitor.LNodes, Visitor.thePathDecl),
        (4, 4, conf[2].body[0]))

    # Getslice
    r = conf[0].body[0:2]
    G.t("Getslice", (len(r) == 2) and (r[0] is conf[0].body[0]) and
        (r[1] is conf[0].body[1]))
    conf[0].body[1:1] = [ "A=B\n", "C=D\n" ]
    G.t("Setslice", conf[0].body.dump(0),
        "k=v a z\n    A=B\n    C=D\n    frob w = s\n    z=path frob\n\n")
    conf[2].body[0].ps.append(":")
    conf[2].body[0].ps.append("a")
    conf[2].body[0].ps.extend([":", "b"])
    conf[2].body[0].ps.insert(0,"p:")
    G.t("Append, extend, insert", conf[2].body[0].ps, "p:x:y CLN z:a:b")
    psNode = kvl.getTypeSystem().newNode("PathSpec", ["a", ":", "b"])
    declNode = kvl.newNode("Decl:decl")
    declNode.key = "Foo"
    declNode.val = kvl.newNode("Val", ["bar"])
    declNode.cleanSpace(kvl.DefaultSpace)
    G.t("newNode with format and type #1", psNode.dump(), "a : b ")
    G.t("newNode with format and type #2", declNode, "Foo = bar \n")
    headNode = kvl.newNode(kvl.getType("Header"))
    hn2 = headNode.clone()
    headNode.k = "Some Enchanted Section"
    headNode[0].setSpace("")
    headNode[1].setSpace("")
    headNode[2].setSpace("")
    G.t("newNode with type and no format", (str(headNode),str(hn2)),
	("[Some Enchanted Section]\n", '[ Frob ] \012'))
    leafNode = kvl.newNode(None, "X")
    mNode = kvl.newNode(None, ["A", "B", "C"])
    mNode.cleanSpace(" ")
    G.t("newNode with no type #1",
        (leafNode.dump(), leafNode.__class__),
        ("X ", parsely.tree.LeafNode)) 
    G.t("newNode with no type #2.1", mNode.dump(), "A B C ")
    G.t("newNode with no type #2.2", mNode[1], "B")
    # This should guess Decl:decl without warning.
    declDecl = kvl.newNode('Decl',val=[None, "k", '=', 'v', '\n'])
    G.t("newNode with uncertain type", str(declDecl.getType()), "Decl:decl")
    #print "This should warn:"
    #declDecl = kvl.newNode('Decl',val=["k", '=', 'v', '\n'])
    #G.t("newNode with no type #3", str(declDecl.getType()), "Decl:decl")

    BOGUS = """
    [sec1]
    k=v = a
    """
    try:
        kvl.parse(BOGUS)
        G.t("Error", 0)
    except parsely.earley_parser.ParseError, msg:
        G.t("Error", str(msg),
            "Syntax error at or near EQ '=' on line 3")        

    ex2 = """
    [foo]
    X1 a frob b frob c 
    X2 z;y;   x;
    """
    conf = kvl.parse(ex2)
    G.t("Parsing with exclusive separators(1)", str(conf), ex2)
    x1s = conf[0].body[0].x1s
    G.t("Exclusive separators(1)", 
	(str(x1s.getType()), x1s.getType().exclusive, str(x1s.__class__)),
	('X1Spec', 1, 'parsely.tree.ExListNode'))
    G.t("Exclusive separators:get", 
	str(conf[0].body[0].x1s[1]), "b")
    G.t("Exclusive separators:getSlice", 
	tuple(map(str, conf[0].body[0].x1s[0:2])), ('a', 'b'))
    x1s[1] = 'foo'
    G.t("Exclusive separators:set", 
	str(x1s), "a frob foo frob c")
    x1s.extend( ["d", "e"] )
    conf.cleanSpace()
    G.t("Exclusive separators:extend",
	(str(x1s), str(x1s[4])), ("a frob foo frob c Frob d Frob e",'e'))
    x1s[2:4] = ['C', 'D']
    G.t("Exclusive separators:setslice",
    	str(x1s), "a frob foo frob C Frob D Frob e")
    del x1s[1]
    del x1s[3]
    G.t("Exclusive separators:delitem",
    	str(x1s), "a frob C Frob D")
    G.t("Exclusive separators:len", len(x1s), 3)
    del x1s[0:1]
    del x1s[1:]
    G.t("Exclusive separators:delslice", str(x1s), "C")
    
    x2s = conf[0].body[1].x2s
    G.t("Exclusive terminators(1)", 
	(str(x2s.getType()), x2s.getType().exclusive, str(x2s.__class__)),
	('X2Spec', 1, 'parsely.tree.ExListNode'))
    G.t("Exclusive terminators:get", 
	str(x2s[1]), "y")
    G.t("Exclusive terminators:getSlice", 
	tuple(map(str, x2s[0:2])), ('z', 'y'))

    x2s[1] = 'foo'
    G.t("Exclusive terminators:set", str(x2s), "z;foo;   x;")
    x2s.extend( ["d", "e"] )
    conf.cleanSpace()
    G.t("Exclusive terminators:extend",
	(str(x2s), str(x2s[4])), ("z;foo;   x;d; e;", 'e'))
    x2s[2:4] = ['C', 'D']
    G.t("Exclusive terminators:setslice",
    	str(x2s), "z;foo;   C ; D ; e;")
    del x2s[1]
    del x2s[3]
    G.t("Exclusive terminators:delitem",
    	str(x2s), "z;C ; D ;")
    G.t("Exclusive terminators:len", len(x2s), 3)
    del x2s[0:1]
    del x2s[1:]
    G.t("Exclusive terminators:delslice", str(x2s), "C ;")

    return G.result()

def testNodeErrors(V=0):
    """Errors in node manipulation"""
    from parsely import *
    
    G = _tGroup("Bad node manipulations", level=1, V=V)
    ex1 = """

    [sec1]

    k=v a z
    frob w = s
    z=path frob

    [sec2]

    [sec3  ]
    path = x:y CLN z
    cmd = x;y;
    """
    kvl = make_simple_grammar()
    conf = kvl.parse(ex1)

    ## Index out of bounds
    try:
        x = conf[-1]
        G.t("Index too low", 0)
    except IndexError, e:
        G.t("Index too low", str(e), "Index out of bounds")
    try:
        x = conf[3]
        G.t("Index too high", 0)
    except IndexError, e:
        G.t("Index too high", str(e), "Index out of bounds")
    try:
        conf[-1] = "hello"
        G.t("Index too low (set)", 0)
    except IndexError, e:
        G.t("Index too low (set)", str(e), "Index out of bounds")
    try:
        x = conf[3]
        G.t("Index too high (set)", 0)
    except IndexError, e:
        G.t("Index too high (set)", str(e), "Index out of bounds")
    ## Attr
    try:
        x = conf.ooop
        G.t("Bad getattr (1)", 0)
    except AttributeError, e:
        G.t("Bad getattr (1)", str(e), "ooop")
    try:
        x = conf[0].nonexist
        G.t("Bad getattr (2)", 0)
    except AttributeError, e:
        G.t("Bad getattr (2)", str(e), "nonexist")
    try:
        x = conf[0].body.nonexist
        G.t("Bad getattr (3)", 0)
    except AttributeError, e:
        G.t("Bad getattr (3)", str(e), "nonexist")
    try:
        conf[0].nonexist = None
        G.t("Bad setattr (1)", 0)
    except AttributeError, e:
        G.t("Bad setattr (1)", str(e), "nonexist")
    try:
        conf[0].body.ooop = None
        G.t("Bad setattr (2)", 0)
    except AttributeError, e:
        G.t("Bad setattr (2)", str(e), "ooop")
    try:
        conf.ooop = None
        G.t("Bad setattr (3)", 0)
    except AttributeError, e:
        G.t("Bad setattr (3)", str(e), "ooop")
    ## Bad index type
    try:
        x = conf["Hello"]
        G.t("Bad index type", 0)
    except TypeError, e:
        G.t("Bad index type", str(e), "Index not an integer" )
    try:
        conf["Hello"] = None
        G.t("Bad index type (set)", 0)
    except TypeError, e:
        G.t("Bad index type (set)", str(e), "Index not an integer" )
    ## Insert mistyped node.
    try:
        conf[0].head.k = 9L
        G.t("Mistyped node (set)", 0)
    except TypeError, e:
        G.t("Mistyped node (set)", str(e),"Got '9L'; expected String or Node")
    try:
        conf[0] = 9L
        G.t("Mistyped node (set2)", 0)
    except TypeError, e:
        G.t("Mistyped node (set2)",str(e),"Got '9L'; expected String or Node")
    try:
        conf.append(9L)
        G.t("Mistyped node (list)", 0)
    except TypeError, e:
        G.t("Mistyped node (list)",str(e),"Got '9L'; expected String or Node")
    try:
        conf.extend([9L, 10L])
        G.t("Mistyped node (list2)", 0)
    except TypeError, e:
        G.t("Mistyped node (list2)",str(e),"Got '9L'; expected String or Node")

    return G.result()
        
def testBadGrammars(V=0):
    """Grammars with errors"""
    from parsely.format import *
    G = _tGroup("Error messages on malformed grammars", level=1,V=V)
    try:
        gram = FileFormat(
            Token("Tok", "SS"))
        gram.process()
        G.t("Missing RE/LIT on token", 0)
    except GrammarError, msg:
        G.t("Missing RE/LIT on token", str(msg),
            "SS is not a valid pattern specifier")
    try:
        gram = FileFormat(
            Space("SS"))
        gram.process()
        G.t("Missing RE/LIT on space", 0)
    except GrammarError, msg:
        G.t("Missing RE/LIT on space", str(msg),
            "SS is not a valid pattern specifier")
    try:
        gram = FileFormat(
            Pattern("PP", "SS"))
        gram.process()
        G.t("Missing RE/LIT on pattern", 0)
    except GrammarError, msg:
        G.t("Missing RE/LIT on pattern", str(msg),
            "SS is not a valid pattern specifier")
    try:
        gram = FileFormat(
            Rule("X", "Y"))
        gram.process()
        G.t("Missing SEQ 1", 0)
    except GrammarError, msg:
        G.t("Missing SEQ 1", str(msg), "Y is not a valid rule specifier")
    try:
        gram = FileFormat(
            Rule("X", ALT("Y", "Z")))
        G.t("Missing SEQ 2", 1)
    except GrammarError, msg:
        G.t("Missing SEQ 2", 0)
    try:
        gram = FileFormat(
            Token("V", RE("{:Nonesuch:}")),
            Start("Foo"),
            Rule("Foo", SEQ("V")))
        gram.process()
        G.t("Bad pattern reference", 0)
    except parsely.remanip.REError, msg:
        G.t("Bad pattern reference", str(msg),
            'No declaration for pattern reference Nonesuch')
    try:
        gram = FileFormat(
            Token("V", RE("x"), inStates=('s1')))
        gram.process()
        G.t("Bad state reference 1", 0)
    except GrammarError, msg:
        G.t("Bad state reference 1", str(msg), 'No such state as s1')
    try:
        gram = FileFormat(
            Token("V", RE("x"), toState=('s1')))
        gram.process()
        G.t("Bad state reference 2", 0)
    except GrammarError, msg:
        G.t("Bad state reference 2", str(msg), 'No such state as s1')
    try:
        gram = FileFormat(
            Space(RE("ABC"), inStates='s2'))
        gram.process()
        G.t("Bad state reference 3", 0)
    except GrammarError, msg:
        G.t("Bad state reference 3", str(msg), 'No such state as s2')
    try:
        gram = FileFormat(
            Space(RE("ABC"), inStates='s2'))
        gram.process()
        G.t("Bad state reference 3", 0)
    except GrammarError, msg:
        G.t("Bad state reference 3", str(msg), 'No such state as s2')
    try:
        gram = FileFormat(
            Rule("V", SEQ('T')),
            Start("V"))
        gram.process()
        G.t("Bad rule reference 1", 0)
    except GrammarError, msg:
        G.t("Bad rule reference 1", str(msg), 'Unrecognized symbol: T')
    try:
        gram = FileFormat(
            Rule("V", SEQ()),
            Start("P"))
        gram.process()
        G.t("Bad rule reference 2", 0)
    except GrammarError, msg:
        G.t("Bad rule reference 2", str(msg), 'Unrecognized symbol: P')
    try:
        gram = FileFormat()
        gram.process()
        G.t("Missing start symbol", 0)
    except GrammarError, msg:
        G.t("Missing start symbol", str(msg), 'No start symbol given')
    try:
        gram = FileFormat(
            State("S1"),
            State("S1"))
        gram.process()
        G.t("Duplicate states", 0)
    except GrammarError, msg:
        G.t("Duplicate states", str(msg), 'Duplicate key S1')
    try:
        gram = FileFormat(
            Pattern("P1", RE("ABC")),
            Pattern("P1", RE("ABC")))
        gram.process()
        G.t("Duplicate patterns", 0)
    except GrammarError, msg:
        G.t("Duplicate patterns", str(msg), 'Duplicate key P1')

    try:
        gram = FileFormat(
            Token("T1", RE("X")),
            Rule("R1", SEQ("T1")),
            Rule("R1", SEQ("T1")))
        gram.process()
        G.t("Duplicate rules", 0)
    except GrammarError, msg:
        G.t("Duplicate rules", str(msg), 'Duplicate key R1')
    return G.result()

def make_action_grammars():
    from parsely.format import *
    # Uses:  MATCH, TYPE, LEN, LINE, ERROR, REINTERPRET
    wCounter = FileFormat(
        Token('wTok', LIT('w'), action='countw'),                
        Space(LIT("\n")),
        Token("Horatio", LIT("Horatio")),
        Rule("HoratioL", SEQ("Horatio")),
	Token("Afloat", RE("alias_\w+"), action="alias"),
	Token("Float", RE("\d+\.\d+")),
	Token("ID", RE("\w+"), action='isDoubleYou'),
	Space(RE("\.\d+"), action="floatError"),
        Space(RE(".")),

        Start("HoratioL"),
	Action('initialize', 'python',
	       """
	       self.wCount = 0
	       self.amDone = 0
	       """,
	       'initScan'),
	
        Action('finish', 'python',
	       """
	       self.amDone = 1
	       """,
	       'finishScan'),

	Action('incrW', 'python',
	       """
	       self.wCount = self.wCount + 1
	       """,
	       'scanFn'),

        Action('countw', 'python',
"""
self.incrW()
self.wMatch = self.match
self.wType = self.type
self.wLen = self.matchLen
self.wLine = self.getLineNumber()
"""
               ),
        Action('isDoubleYou', 'python',
"""
if self.match == 'doubleyou':
   self.setType("wTok")
   self.incrW()
"""
               ),
        Action('floatError', 'python',
"""
self.error("Bad float.", "FATAL")
"""
               ),
        Action('alias', 'python',
"""
  # String is of format alias_\w+.
  word = self.match[6:]
  self.reinterpret("ID", word, "Float")
"""
               )
        )

    # Uses: ACCUMULATE, LESS, BEGIN, MORE
    regexenEtc = FileFormat(        
        State("RE", exclusive=1, noSpace=1),
        State("RE_END", exclusive=1, noSpace=1),
        Space(RE(r"[\s\n]+")),
        Token("reBegin", RE(r"re\W"), toState="RE", action='beginRE'),
        Token("reBody", RE(r"\\."), inStates="RE", action='moreRE'),
        Token("reBody2", RE(r"."), inStates="RE", action='bodyRE'),
        Token("reClose", RE(r'.|\n'), inStates="RE_END", action='endRE'),
        Token("IfYouDontSeeTheFnords", LIT('fnord'), action="trySpace"),
        Token("fOrFoo", LIT('f'), action='sawF'),
        Token("ID", RE(r'\w+')),

        Action('trySpace', 'python', "self.becomeSpace()"),
        
        Action('beginRE', 'python',
"""
close = self.match[2]
if close == '{': close = '}'
if close == '(': close = ')'
if close == '[': close = ']'
if close == '<': close = '>'
self.close=close
self.depth=0
self.inClass=0
"""),
        Action('moreRE', 'python', "ACCUMULATE()"),
        Action('bodyRE', 'python',
"""
if self.inClass:
    if self.match == ']': self.inClass = 0
    self.accumulate()
    CONTINUE

if self.match == '[': self.inClass = 1
if self.match in '{(': self.depth = self.depth + 1
if (self.depth == 0) and self.match == self.close:
    self.enterState("RE_END")
if self.match in '})': self.depth = self.depth - 1
self.accumulate()
"""),
        Action('endRE', 'python',
"""
if self.match not in 'ix': 
    self.less(1)
    self.accumulate()
    self.enterState("INITIAL")
"""),
        
        Start("File"),
        Rule("File", SEQ("reBegin")),

        Action('sawF', 'python',
"""
if len(self.rest()) >= 2 and self.rest()[:2] == 'oo':
    self.more(2)
""")
        )

    parenCounter = FileFormat(
        Space(RE(r'[ \t\n]+')),
        Token('OPEN', LIT("(")),
        Token('CLOSE', LIT(")")),
        Token('ID', RE("\w+")),
        Start("PExpr"),
        Rule("PExpr", SEQ("OPEN", "List", "CLOSE", action="incrCount")),
        Rule("List", MULTI("IdOrPexpr", "+")),
        Rule("IdOrPexpr", ALT("ID", "PExpr", "BadPExpr")),
        Rule("BadPExpr", SEQ("OPEN", "CLOSE",action='gripe')),

        Action("gripe", 'Python', 'self.error("Foo bar!")'),
      
        Action("initialialize", 'Python',
 	 """
	 self.count = 0
	 self.done = 0
	 """,
	kind="initParse"),
	
	Action('finish', 'python', "self.done = 1", 'finishParse'),

	Action('firstM', 'python', 
	 """
	 self.firstMatch = str(self.match)
	 self.firstPartOfFirstMatch = str(self.args[0])
	 self.lineOfFirstMatch = self.getLineNumber()
         """, 'parseFn'),

        Action("incrCount", 'Python',
         """
         self.count = self.count + 1
	 if self.count == 1: self.firstM()
	 """
	 )
        )
        

    wCounter.process()
    regexenEtc.process()
    parenCounter.process()    
    
    return wCounter, regexenEtc, parenCounter

def testActions(V=0):
    """Grammars with actions"""
    G = _tGroup("Grammars with actions", level=1,V=V)
    wCounter, regexen, parenCounter = make_action_grammars()
    
    ex1 = """AWord alias_AWord AWord
             doubleyou
             I hereby proclaim:
	     waxen woad wanes woefully"""
    tok,ctxt = wCounter.scanner.tokenize(ex1,1)
    G.t("Simple scanning actions", (ctxt.wCount,
                                    ctxt.wMatch,
                                    str(ctxt.wType),
                                    ctxt.wLen,
                                    ctxt.wLine,
				    ctxt.amDone),
                                   (5, "w", "wTok", 1, 4, 1))

    G.t("Reinterpret",
	(str(tok[1].val._type),
	 str(tok[2].val._type),
	 str(tok[3].val._type),
	 str(tok[3].val),
	 ), ("ID", "Afloat", "Float", "AWord"))
    G.t("SetType",
	(str(tok[4].val._type), str(tok[4].val)),
	("wTok", "doubleyou"))

    exErr = "This has a bad float: .999 ww"
    try:
	tok = wCounter.scanner.tokenize(exErr)
	G.t("Error actions", 0)
    except parsely.pcre_scanner.ScanError, msg:
	G.t("Error actions", str(msg), "Bad float.")

    ex2 = """  re{abc} re/a(b/)c/ 
               re<a[b>]c>ix f fo foo """
    tok = regexen.scanner.tokenize(ex2)
    G.t("More complex actions #1", (str(tok[1].val),
                                 str(tok[2].val),
                                 str(tok[3].val),
                                 str(tok[4].val),
                                 str(tok[5].val)),
        ("re{abc}", "re/a(b/)c/", "re<a[b>]c>", "i", "x"))
    G.t("More complex actions #2", (str(tok[9].val),
                                    str(tok[6].val._type),
                                    str(tok[7].val._type),
                                    str(tok[8].val._type),
                                    str(tok[9].val._type)),
        ("foo", "fOrFoo", "fOrFoo", "ID", "fOrFoo"))        
    rejoined = tok[0]
    for t in tok[1:]:
        rejoined = rejoined + t.val.dump(1)
    G.t("More complex actions #3", rejoined, ex2)

    tok = regexen.scanner.tokenize("fnord fnord fnord")
    G.t("SPACE action", 1, len(tok))

    ex3 = """ ( this has\n( a total of) (three parenthesized expressions))"""
    tree,c = parenCounter.parser.parse(parenCounter.scanner.tokenize(ex3),1)

    G.t("Simple parser actions",
        (c.firstMatch, c.firstPartOfFirstMatch, c.lineOfFirstMatch, c.count,
	 c.done),
        ('(three parenthesized expressions)', '(', 2, 3, 1))
    try:
        parenCounter.parse("((()))")
        G.t("Parser errors", 1)
    except parsely.earley_parser.ParseError, msg:
        G.t("Parser errors", str(msg),
            "Foo bar! at or near BadPExpr '()'")

    return G.result()
                      
######################################################################
## File manipulation
######################################################################

def getFormats(name):
    comp = name + ".plyC"
    if os.path.exists(comp):
	os.remove(comp)

    fmt1 = parsely.loadFormat(name)
    compiled = os.path.exists(comp)
    fmt2 = parsely.loadFormat(name)

    return fmt1 , compiled, fmt2

def testLoadFormat(V=0):
    """Loading/compiling file formats."""
    assert 0
    # This function is disabled, until format loading is a going
    # concern again.

    V=1 #XXXX
    G = _tGroup("Loading/compiling file formats", level=2,V=V)

    #a1_1,a1_c,a1_2 = getFormats("tests/actions1")
    #G.t("actions1 compiled", a1_c)

    a= getFormats("tests/actions3")

    import pickle
    for i in a.ruleList[2].val.alternatives.keys():
	print i
	m = a.ruleList[2].val.alternatives[i]
	print m, m.__dict__
	ste = m.__getstate__()
	seq = parsely.format.SEQ()
	seq.__setstate__(ste)
	#for k,v in m.__dict__.items():
	#    print k
	#    pickle.loads(pickle.dumps(v))

	print m.type, m.type.__dict__
	for mem in m.type.defaultMembers:
	    print repr(mem)
	    pickle.loads(pickle.dumps(mem))
	    print "DONE"

	print "#"
	pickle.loads(pickle.dumps(m.type))
	

	print "?"
	pickle.loads(pickle.dumps(m))
	print "!"

    
    #a3_1,a3_c,a3_2 = getFormats("tests/actions3")

    #G.t("actions3 compiled", a3_c)   

    return G.result()

def testFileReplacement(V=0):
    """Replacing files"""
    G = _tGroup("Replacing files", level=2,V=V)

    from parsely._util import fileContents
    import parsely.grammar

    try:
	fmt = parsely.grammar.generateFileFormat(
	    fileContents("tests/basic_kvl.ply"))
	fname = '/tmp/test_parsely_tmp_' + str(os.getpid())
	f = open(fname, 'w')
	f.write("A=alpha\nB=Beta\nZ=zeta\n\n")
	f.close()
	node = fmt.parseFile(fname)
	node.body[1].v = 'Betelgeuse'
	node.flush()
	del node
	val = fileContents(fname)
	G.t("Replacing single kvl file", val,
	    "A=alpha\nB=Betelgeuse\nZ=zeta\n\n")
    finally:
	try:
	    os.unlink(fname)
	except:
	    pass

    return G.result()

######################################################################
## Helper functions
######################################################################

def writeRes(name, ok, level):
    if ok:
	print "  " * level + "OK ", name
    else:
	print "  " * level + "BAD", name

class _tGroup:
    def __init__(self, name, level=1,V=0):
        self.banner = ("  " * (level-1)) + "--- " + name
        if V>0:
            print self.banner
            self.banner = None
	self.name = name
	self.ok = 0
	self.bug = 0
        self.V=V
	self.level = level
    def t(self,descr,val,expected=None):
        if expected is not None:
            ok = (val == expected)
        else:
            ok = val
        if (not ok) and self.banner:
            print self.banner
            self.banner = None
        if (not ok) or self.V:
            writeRes(descr,ok,self.level)
        if (not ok) and expected is not None:
            print ("  " * (self.level+2)) + "EXPECTED ", expected
            print ("  " * (self.level+2)) + "     GOT ", val            
        if ok:
            self.ok = self.ok+1
        else:
            self.bug = self.bug+1
    def test(self, f, args):        
	o,b = _tFn("*", f,args,self.level,self.V,P=0)
        doc = f.__doc__
        self.t(doc, o)
    def result(self):
	return (self.ok, self.bug)

def _runSequence(name, cmds, argl, level=1):
    G = _tGroup(name,level)
    for c in cmds:
	G.t(c,argl)
    return G.result()

def _tFn(name, fn, args = (), level = 0,V=0,P=1):
    doc = name
    if fn.__doc__: doc = "%s (%s)" % (fn.__doc__, name)
    if level==0 and not V:
        print "*", doc
    try:
        if V:
            success = apply(fn,args,{'V':1})
        else:
            success = apply(fn,args)
	if isinstance(success,types.TupleType):
	    return success
	elif success:
            if V and P:
                writeRes(doc, 1, level)
	    return (1,0)
	else:
            if P:
                writeRes(doc, 0, level)
	    return (0,1)
    except "A":
	pass
#	print "EXC", doc
#	return (0,1)

def run():
    import sys
    global V
    V = 0
    if len(sys.argv) >1 and sys.argv[1] == '-v':
        V = 1
    bug = 0
    ok = 0
    for fn in (testReManip, testReErrors, testPCRELex, testNoLexingHack,
	       testParserAndNodes,
	       testNodeErrors, testBadGrammars, testActions,
	       testFileReplacement ):
	o,b = _tFn(fn.__name__,fn,V=V)
        ok = ok + o
        bug = bug + b

    print bug, "tests failed"
    print ok, "tests passed"
    
if __name__ == '__main__':
    run()

